!!!
! Testing random number generators provided by Numerical Recipes book (2nd ed.)
!  author: Gabriele Sclauzero (EPFL, Lausanne), Sept. 2010 
!!!
program test_ran

implicit none
integer :: iran, ntry, nbins, nexp

integer, parameter :: nbinx=1000, nbin2=50 ! max number of histogram bins
real, parameter :: chisqx=40.0, chisq2dx=200.0 ! max chisq in histogram
integer :: iexp, itry, i, j, seed, t(9)
integer :: cnt(nbinx), cnt0(nbinx), distr(nbin2), nchisq
integer :: cnt2d(nbinx,nbinx), cnt2d0(nbinx,nbinx), distr2d(nbin2), nchisq2d
real :: res, resold, binw, chisq, invdchisq, invdchisq2d, nideal, intdistr(nbin2) 
character(len=256) :: filename
character(len=16) :: iranc, nexpc, ntryc, nbinsc, date, time, zone 

!!! RNG from external functions
real :: ran0, ran1, ran2, ran3
external :: ran0, ran1, ran2, ran3

write(*,'(A)',advance='NO') "iran:"
read(*,'(I22)') iran 
write(*,'(A)',advance='NO') "number of tries:"
read(*,'(I22)') ntry
write(*,'(A)',advance='NO') "number of bins:"
read(*,'(I22)') nbins
write(*,'(A)',advance='NO') "number of experiments:"
read(*,'(I22)') nexp

if ( ntry <= 0 .OR. nbins <= 0 .OR. nexp <= 0 ) then
   write(*,'(A)') "wrong input"
   stop
endif
if ( nbins > nbinx ) then
   write(*,'(A,I22)') "maximum number of bins=",nbinx
   stop
endif

!! fixed seed (for debugging purposes)
!seed = 123456789
!! random seed 
call date_and_time(date,time,zone,t) 
!seed = t(8)+1000*(t(7) + 60*(t(6) + 60*(t(5) + 24*(t(3) - 1 + 31*(t(2) - 1 +12*t(1))))))
!! the version above generates overflows... some compilers don't like
seed = t(8)+1000*(t(7) + 60*(t(6) + 60*(t(5))))
!seed = imsec + 60*(imin + 60*(ihr + 24*(iday - 1 + 31*(imon - 1 +12*iyr))))
seed = ior(seed,1) ! to ensure that seed is an odd number


cnt(:) = 0
cnt0(:) = 0
distr(:) = 0
cnt2d(:,:) = 0
cnt2d0(:,:) = 0
distr2d(:) = 0

binw = 1.0 / real(nbins) 
invdchisq = real(nbin2) / chisqx
invdchisq2d = real(nbin2) / chisq2dx
nchisq = 0
nchisq2d = 0

do iexp = 1,nexp

   do itry = 1,ntry

      selectcase ( iran )
      case ( 0 )
         res = ran0(seed)
      case ( 1 ) 
         res = ran1(seed)
      case ( 2 ) 
         res = ran2(seed)
      case ( 3 ) 
         res = ran3(seed)
      case default
         write(*,'(A,I22)') "unknown generator ran",iran
         stop
      endselect

      call binning(res)

      if ( mod(itry,2) == 0 ) call binning2d(res,resold)
      resold = res

   enddo 

   ! compute X^2 for R_{i}
   chisq = 0.0
   nideal = ntry/real(nbins)
   do i = 1, nbins
      chisq = chisq + (cnt(i) - cnt0(i) - nideal)**2
      cnt0(i) = cnt(i)
   enddo
   chisq = chisq / nideal

   call chisq_distr(chisq)
   !write(*,'(A,I8,A,F14.6)') "Exp #",iexp," :X^2 =",chisq

   ! compute X^2 for (R_{i},R_{i-1})
   chisq = 0.0
   nideal = ntry/2.0/real(nbins)**2
   do i = 1, nbins
      do j = 1, nbins
         chisq = chisq + (cnt2d(i,j) - cnt2d0(i,j) - nideal)**2 
         cnt2d0(i,j) = cnt2d(i,j)
      enddo
   enddo
   chisq = chisq / nideal

   call chisq_distr2d(chisq)
   !write(*,'(A,I8,A,F14.6)') "Exp #",iexp," :X^2 2D =",chisq


enddo


! write histogram to file
write(nexpc,'(I16)') nexp
write(ntryc,'(I16)') ntry
write(nbinsc,'(I5)') nbins 
write(iranc,'(I1)') iran
filename = 'histo-ran'//trim(adjustl(iranc))// '_'// &
 trim(adjustl(nexpc))//'x'//trim(adjustl(ntryc))// '-'//trim(adjustl(nbinsc)) 

open(2,file=trim(filename),action="write",status="unknown")
write(2,'(2F14.6)') ( binw*(real(i)-0.5), &
   cnt(i)/(real(nexp)*real(ntry))*real(nbins), i=1,nbins )
close(2)

if ( nchisq > nbin2 ) then

   ! write the X^2 distribution to file
   filename = 'chisq-ran'//trim(adjustl(iranc))// '_'// &
    trim(adjustl(nexpc))//'x'//trim(adjustl(ntryc))// '-'//trim(adjustl(nbinsc)) 

   ! compute the integral of the X^2 distribution
   intdistr(1) = distr(1)/2.0
   do i=2,nbin2
      intdistr(i) = intdistr(i-1) + (distr(i-1) + distr(i))/2.0
   enddo

   open(2,file=trim(filename),action="write",status="unknown")
   write(2,'(3F14.6)') ( (real(i)-0.5)/invdchisq, &
      distr(i)*invdchisq/real(nexp), intdistr(i)/real(nexp), i=1,nbin2 )
   close(2)

   ! write the 2D X^2 distribution to file
   filename = 'chisq2d-ran'//trim(adjustl(iranc))// '_'// &
    trim(adjustl(nexpc))//'x'//trim(adjustl(ntryc))// '-'//trim(adjustl(nbinsc)) 

   ! compute the integral of the 2D X^2 distribution
   intdistr(1) = distr2d(1)/2.0
   do i=2,nbin2
      intdistr(i) = intdistr(i-1) + (distr2d(i-1) + distr2d(i))/2.0
   enddo

   open(2,file=trim(filename),action="write",status="unknown")
   write(2,'(3F14.6)') ( (real(i)-0.5)/invdchisq2d, &
      distr2d(i)*invdchisq2d/real(nexp), intdistr(i)/real(nexp), i=1,nbin2 )
   close(2)

else
   write(*,'(A,I22)') "too few X^2 values: try to increase nexp", nchisq
endif

!!!
contains
!!!


subroutine binning(r)

real :: r

integer :: ibin

! assumes 0 <= r < 1
ibin = int(real(nbins)*r) + 1
cnt(ibin) = cnt(ibin) + 1

end subroutine


subroutine binning2d(r1,r2)

real :: r1,r2

integer :: ibin1, ibin2

! assumes 0 <= r < 1
ibin1 = int(real(nbins)*r1) + 1
ibin2 = int(real(nbins)*r2) + 1
cnt2d(ibin1,ibin2) = cnt2d(ibin1,ibin2) + 1

end subroutine


subroutine chisq_distr(x2)

real :: x2

integer :: ibin

if ( x2 < chisqx ) then
   ibin = int(invdchisq*x2) + 1
   distr(ibin) = distr(ibin) + 1
   nchisq = nchisq + 1
endif

end subroutine


subroutine chisq_distr2d(x2)

real :: x2

integer :: ibin

if ( x2 < chisq2dx ) then
   ibin = int(invdchisq2d*x2) + 1
   distr2d(ibin) = distr2d(ibin) + 1
   nchisq2d = nchisq2d + 1
endif

end subroutine

end program
