!!!
! Testing correlations in random number generators 
! (ran0, ran1, ran2, ... provided by Numerical Recipes book, 2nd ed.)
!  author: Gabriele Sclauzero (EPFL, Lausanne), Sept. 2010 
!!!
program test_corr

implicit none
integer :: iran, ntry, ndim

integer, parameter :: ndimx=9, nbins=100
! beware, if you increase ndimx you must change the output format string
integer :: i, j, ibuf, seed, t(9)
integer :: cnt_prod(nbins), cnt_diff(nbins), cntlow ! variables for histograms
integer :: idum, ia,im,ic,i0 ! parameters for simple LCG
real :: invm
real :: res, buf(ndimx), binw, rprod, rdiff, resold, rlowav
character(len=256) :: filename
character(len=16) :: iranc, ndimc, ntryc, date, time, zone
! trick to overcome bogus(?) gfortran compilers
integer(8), parameter :: two=2

!!! RNG from external functions
real :: ran0, ran1, ran2, ran3
external :: ran0, ran1, ran2, ran3

write(*,'(A)',advance='NO') "iran:"
read(*,'(I22)') iran 
if ( iran == 9 ) then
   write(*,'(A)',advance='NO') "parameters of the LCG (a,c,m,i0):"
   !read(*,'(4I22)') ia,ic,im,i0 ! this doesn't work on some compilers
   read(*,*) ia,ic,im,i0
   do while ( dble(ia)*dble(im)+dble(ic) > dble(two**31-1) )
      write(*,'(A)') &
         "this generator does not avoid overflow, chose smaller parameters"
      write(*,'(A)',advance='NO') "parameters of the LCG (a,c,m,i0):"
      read(*,'(4I22)') ia,ic,im,i0 
   enddo
   invm = 1./im
endif

write(*,'(A)',advance='NO') "number of tries:"
read(*,'(I22)') ntry
write(*,'(A)',advance='NO') "number of dim.:"
read(*,'(I22)') ndim

if ( ntry <= 0 .OR. ndim <= 0 ) then
   write(*,'(A)') "wrong input"
   stop
endif

if ( ndim > ndimx ) then
   write(*,'(A,I22)') "maximum number of dimensions =",ndimx
   stop
endif

!! fixed seed (for debugging purposes)
!seed = 123456789
!! random seed 
call date_and_time(date,time,zone,t) 
!seed = t(8)+1000*(t(7) + 60*(t(6) + 60*(t(5) + 24*(t(3) - 1 + 31*(t(2) - 1 +12*t(1))))))
!! the version above generates overflows... some compilers don't like
seed = t(8)+1000*(t(7) + 60*(t(6) + 60*(t(5))))
!seed = imsec + 60*(imin + 60*(ihr + 24*(iday - 1 + 31*(imon - 1 +12*iyr))))
seed = ior(seed,1) ! to ensure that seed is an odd number
if ( iran == 9 ) seed = i0 ! user supplied seed for LCG

binw = 1.0 / real(nbins) 

! write n-dimensional data to file
write(ntryc,'(I16)') ntry
write(ndimc,'(I1)') ndim 
write(iranc,'(I1)') iran
filename = 'corr-ran'//trim(adjustl(iranc))//'_'// &
 trim(adjustl(ndimc))//'dim-'//trim(adjustl(ntryc))
open(2,file=trim(filename),action="write",status="unknown")

cnt_prod(:) = 0
cnt_diff(:) = 0
cntlow = 0
rlowav = 0.d0
j=0

do i = 1,ntry

   selectcase ( iran )
   case ( 0 )
      res = ran0(seed)
   case ( 1 ) 
      res = ran1(seed)
   case ( 2 ) 
      res = ran2(seed)
   case ( 3 ) 
      res = ran3(seed)
   case ( 9 ) 
      res = lcg(seed)
   case default
      write(*,'(A,I22)') "unknown generator ran",iran
      stop
   endselect

   if ( i > 1 ) then
      !
      ! Test #1: distribution of R_{i-1}*R_{i} and R_{i-1}-R_{i}
      rprod = resold * res
      rdiff = resold - res
      call binning(rprod,rdiff)
      !
      ! Test #2: average of numbers following a very small R.N. 
      if ( resold < 1.d-5 ) then
         cntlow = cntlow + 1
         rlowav = rlowav + res
      endif
      !
   endif

   resold = res

   ! Test #3: n-dimensional correlation   
   !if ( res > 0.0005 ) cycle
   j = j+1
   ibuf = mod(j,ndim)
   if ( ibuf == 0 ) then
      buf(ndim) = res
      write(2,'(9F14.6)') buf(1:ndim)
   else
      buf(ibuf) = res
   endif

enddo 

close(2)

! write histogram of the distribution of the product to file
filename = 'distprod-ran'//trim(adjustl(iranc))//'_'//trim(adjustl(ntryc)) 

open(2,file=trim(filename),action="write",status="unknown")
write(2,'(4F14.6)') ( &
  binw*(real(i)-0.5), cnt_prod(i)/real(ntry)*real(nbins),&
  binw*(real(i)-0.5)*2.0 - 1.0, cnt_diff(i)/real(ntry)*real(nbins),&
  i=1,nbins )
close(2)

! write on output the average 
if ( cntlow > 0 ) then
   write(*,'(A,I6,A,F14.9)') &
      "average after R.N.<10^-5 (",cntlow," events):", rlowav/real(cntlow)
else
   write(*,'(A)') "no events with R.N.<10^-5: increase ntry"
endif


!!!
contains
!!!


subroutine binning(rp,rd)

implicit none
real :: rp, rd

integer :: ibin

! binning of the product
! assumes 0 <= r < 1
ibin = int(real(nbins)*rp) + 1
cnt_prod(ibin) = cnt_prod(ibin) + 1
! binning of the difference 
! assumes -1 < r < 1
ibin = int(real(nbins)*(rd+1.d0)/2.d0) + 1
cnt_diff(ibin) = cnt_diff(ibin) + 1

end subroutine


function lcg(idum)
!!!
! simple LCG without overflow avoiding mechanism and without "mask"
!!!
implicit none
integer idum
real lcg

idum = mod(ia*idum + ic, im)
lcg = idum*invm

end function

end program test_corr
