"""
This file contains the specific functions to solve Maxwell equations
 using a WENO scheme.
"""

import numpy as np
from Maxwell import *
from Weno import *
from helpers import extend


def MaxwellWENOrhs1D(x,q,ep,mu,h,k,m,Crec,dw,beta,maxvel):

    """Purpose: Evaluate the RHS of Maxwell equations using a WENO reconstruction"""
    N = len(x)
    dq = np.zeros((2,N))

    qe = np.zeros((2,N+2*m))

    # Extend data and assign boundary conditions
    xe,qe[0,:] = extend(x, q[0,:], m, "D", 0.0, "D", 0.0)
    xe,qe[1,:] = extend(x, q[1,:], m, "N", 0, "N", 0)

    #define cell left and right interface values
    ql = np.zeros((2,N+2))
    qr = np.zeros((2,N+2))

    for i in range(N+2):
        ql[0,i], qr[0,i] = WENO(xe[i:1+(i+2*(m-1))],qe[0,i:1+(i+2*(m-1))],m,Crec,dw,beta)
        ql[1,i], qr[1,i] = WENO(xe[i:1+(i+2*(m-1))],qe[1,i:1+(i+2*(m-1))],m,Crec,dw,beta)

    # Lax-Friedrichs flux or something simple
    dq = - (MaxwellLF(qr[:,1:N+1], ql[:,2:N+2], ep, mu, maxvel) - \
            MaxwellLF(qr[:,:N], ql[:,1:N+1], ep, mu, maxvel))/h

    return dq

def MaxwellWENO1D(x,q,ep,mu,h,m,CFL,FinalTime):
    """Purpose: Integrate 1D Maxwell equation until FinalTime using a WENO
       scheme and 3rd order SSP-RK method
    """   

    t = 0.0
    tstep = 0

    #Initialize reconstruction weights
    Crec = np.zeros((m+1,m))
    for r in range(-1,m):
        Crec[r+1,:] = ReconstructWeights(m,r)

    # Initialize linear weights
    dw = LinearWeights(m,0)

    # Compute smoothness indicator matrices
    beta = np.zeros((m,m,m))
    for r in range(m):
        xl = -1/2 + np.arange(-r,m-r+1)
        beta[:,:,r] = betarcalc(xl,m)

    cvel = 1.0/np.sqrt(ep*mu)
    maxvel = cvel.max()
    k = CFL*h/maxvel

    # Integrate scheme
    while (t<FinalTime):
        #Decide on timestep
        k = min(k,FinalTime-t)
        #Update solution
        rhsq  = MaxwellWENOrhs1D(x, q,ep,mu,h,k,m,Crec,dw,beta,maxvel)
        q1 = q + k*rhsq
        rhsq  = MaxwellWENOrhs1D(x, q1,ep,mu,h,k,m,Crec,dw,beta,maxvel) 
        q2 = (3*q + q1 + k*rhsq)/4
        rhsq  = MaxwellWENOrhs1D(x, q2,ep,mu,h,k,m,Crec,dw,beta,maxvel)
        q = (q + 2*q2 + 2*k*rhsq)/3
        
        t += k
        tstep += 1

    return q
