"""
This file contains the specific functions to solve Maxwell equations
in 1D using a spectral approach.
"""

import numpy as np
from Maxwell import *
from Spectral import *

def MaxwellSpecrhs1D(x,q,ep,mu,L):

    """Purpose  : Evaluate the RHS of Maxwell equations using spectral Fourier
                  collocation method"""

    N = len(x)
    dq = np.zeros((2,N))

    dq[0,:] = 2.0*np.pi/L*Fourierdx(q[1,:],1.0)/ep
    dq[1,:] = 2.0*np.pi/L*Fourierdx(q[0,:],1.0)/mu

    return dq

def MaxwellSpec1D(x,q,ep,mu,N,L,CFL,FinalTime):
    """Purpose  : Integrate 1D Maxwell equation until FinalTime using a Fourier
                  spectral collocation method and 3rd order SSP-RK method
    """   

    t = 0.0
    tstep = 0
    h = L/(2.0*N+1)

    # Set timestep
    cvel = 1.0/np.sqrt(ep*mu)
    k = CFL*h/cvel.max()

    #integrate scheme
    while (t<FinalTime):
        #Decide on timestep
        k = min(k,FinalTime-t)
        # Update solution 
        rhsq  = MaxwellSpecrhs1D(x, q,ep,mu,L)
        q1 = q + k*rhsq
        rhsq  = MaxwellSpecrhs1D(x, q1,ep,mu,L)
        q2 = (3*q + q1 + k*rhsq)/4
        rhsq  = MaxwellSpecrhs1D(x, q2,ep,mu,L)
        q = (q + 2*q2 + 2*k*rhsq)/3
        
        t += k
        tstep += 1

    return q
