"""
This file contains the specific functions to solve Maxwell equations
in 1D or 2D using a monotone scheme.
"""

import numpy as np
from Maxwell import *
from helpers import extend


def MaxwellMrhs1D(x,EM,ep,mu,h,k,maxvel):

    """Purpose: Evaluate right hand side for Maxwell equation using monotone method"""
    N = len(x)
    EMe = np.zeros((2,N+2))

    # PEC boundary conditions by mirrow principle
    xe,EMe[0,:] = extend(x,EM[0,:], 1, "D", 0.0, "D", 0.0)
    xe,EMe[1,:] = extend(x,EM[1,:], 1, "N", 0.0, "N", 0.0)


    # Change numerical flux here
    dEM = - (MaxwellLF(EMe[:,1:N+1], EMe[:,2:N+2], ep,mu, maxvel) - \
            MaxwellLF(EMe[:,:N], EMe[:,1:N+1], ep,mu, maxvel))/h

    return dEM

def MaxwellM1D(x,EM,ep,mu,h,CFL,FinalTime):
    """Purpose: Integrate 1D Maxwell equation until FinalTime using a monotone scheme.
    """   
    t = 0.0
    timestep = 0

    # Set timestep
    cvel = 1.0/np.sqrt(ep*mu)
    maxvel = cvel.max()
    k = CFL*h/maxvel

    while t < FinalTime:
        k = min(FinalTime-t, k)
        # Update solution
        EM += k*MaxwellMrhs1D(x,EM,ep,mu,h,k,maxvel)
        t +=k
        timestep += 1
        
    return EM

