"""
This file contains the specific functions to solve Maxwell equations
in 1D or 2D using a ENO scheme.
"""

import numpy as np
from Maxwell import *
from Eno import *
from helpers import extend

def MaxwellENOrhs1D(x,EM,ep,mu,h,k,m,Crec,maxvel):

    """Purpose: Evaluate the RHS of Maxwell equations using an ENO reconstruction"""
    N = len(x)
    EMe = np.zeros((2,N+2*m))

    # Extend data and assign boundary conditions
    xe,EMe[0,:] = extend(x,EM[0,:], m, "D", 0.0, "D", 0.0)
    xe,EMe[1,:] = extend(x,EM[1,:], m, "N", 0.0, "N", 0.0)

    #define cell left and right interface values
    EMm = np.zeros((2,N+2))
    EMp = np.zeros((2,N+2))

    for i in range(N+2):
        EMm[0,i], EMp[0,i] = ENO(xe[i:1+(i+2*(m-1))],EMe[0,i:1+(i+2*(m-1))],m,Crec)
        EMm[1,i], EMp[1,i] = ENO(xe[i:1+(i+2*(m-1))],EMe[1,i:1+(i+2*(m-1))],m,Crec)

    # # Lax Friedrich flux - or something simple
    # dEM = - (MaxwellLF(EMp[:,1:N+1], EMm[:,2:N+2], maxvel) - \
    #         MaxwellLF(EMp[:,:N], Emm[:,1:N+1], maxvel))/h

    # Exact upwinding
    xe, epe = extend(x, ep, 1, "N", 0.0, "N", 0.0)
    xe, mue = extend(x, mu, 1, "N", 0.0, "N", 0.0)

    dEM = - (MaxwellUpwind(EMp[:,1:N+1], EMm[:,2:N+2], epe[1:N+1], epe[2:N+2], \
                           mue[1:N+1], mue[2:N+2], epe[1:N+1], mue[1:N+1]) - \
              MaxwellUpwind(EMp[:,:N], EMm[:,1:N+1], epe[:N], epe[1:N+1], \
                           mue[:N], mue[1:N+1], epe[1:N+1], mue[1:N+1])  )/h
    return dEM

def MaxwellENO1D(x,q,ep,mu,h,m,CFL,FinalTime):
    """Purpose: Integrate 1D Maxwell equation until FinalTime using a ENO scheme and 3rd order SSP-RK method.
    """   
    t = 0.0
    tstep = 0

    #Initialize reconstruction weights
    Crec = np.zeros((m+1,m))
    for r in range(-1,m):
        Crec[r+1,:] = ReconstructWeights(m,r)

    cvel = 1.0/np.sqrt(ep*mu)
    maxvel = cvel.max()
    k = CFL*h/maxvel

    #integrate scheme
    while (t<FinalTime):

        k = min(k,FinalTime-t)
        #Update solution
        rhsq  = MaxwellENOrhs1D(x,q,ep,mu,h,k,m,Crec,maxvel)
        q1 = q + k*rhsq
        rhsq  = MaxwellENOrhs1D(x,q1,ep,mu,h,k,m,Crec,maxvel) 
        q2 = (3*q + q1 + k*rhsq)/4
        rhsq  = MaxwellENOrhs1D(x,q2,ep,mu,h,k,m,Crec,maxvel)
        q = (q + 2*q2 + 2*k*rhsq)/3
        
        t += k
        tstep += 1

    return q
