"""
This file contains all the basic implementations of functions to compute
quantities for the specific case of the Burgers equations.
"""

import numpy as np

def MaxwellLF(u,v,ep,mu,maxvel):
    """Purpose: Evaluate  Lax Friedrich numerical flux for Maxwell's equations"""

    fu = np.array((u[1,:]/ep, u[0,:]/mu))
    fv = np.array((v[1,:]/ep, v[0,:]/mu))

    flux = 0.5*(fu+fv - maxvel*(v-u))
    return flux 

def MaxwellLW(u,v,ep,mu,maxvel):
    """Purpose: Evaluate  Lax Wendroff numerical flux for Maxwell's equations"""

    fu = np.array((u[1,:]/ep, u[0,:]/mu))
    fv = np.array((v[1,:]/ep, v[0,:]/mu))
    A2 = np.array((1.0/ep*mu, 1.0/ep*mu))

    flux = 0.5*(fu+fv - maxvel*A2*(v-u))
    return flux 

def MaxwellUpwind(u,v,epu,epv,muu,muv,ep,mu):
    """Purpose: Evaluate exact upwind numerical flux for Maxwell's equation with
        discontinuous coefficients"""

    Zm = np.sqrt(muu/epu)
    Zp = np.sqrt(muv/epv)
    Zavg = 0.5*(Zm+Zp)

    Ym = 1.0/Zm
    Yp = 1.0/Zp
    Yavg = 0.5*(Ym+Yp)

    Hs = 0.5*(Zm*u[1,:]+Zp*v[1,:] + u[0,:]-v[0,:])/Zavg
    Es = 0.5*(Ym*u[0,:]+Yp*v[0,:] + u[1,:]-v[1,:])/Yavg

    numflux = np.array(( Hs/ep, Es/mu ) )
    return numflux



