"""
This file contains the specific functions to solve KPP equations
in 2D using a central scheme.
"""

import numpy as np
from KPP import *
from helpers import extend, extendstag, SlopeLimit

## 2D ##
###################################################################################
def KPPC2D(x, y, u, hx, hy, CFL, FinalTime):
    """Purpose: Integrate 2D KPP equation until FinalTime using second order central scheme.
    """   
    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    maxvel = 1.0
    k = CFL*delta/maxvel/2
    
    # Integrate scheme
    while t < FinalTime:
        # Decide on timestep
        k = min(FinalTime-t, k)
        u += k*KPPCrhs2D(x,y,u,hx,hy,k,maxvel)
        t += k
        tstep += 1

    return u

def KPPCrhs2D(x, y, u, hx, hy, k, maxvel):
    """Purpose: Evaluate right hand side for 2D KPP equation 
        using second order central scheme
    """ 

    Ny,Nx = u.shape

    ue = np.zeros((Ny+4,Nx+4))

    duLx = np.zeros((Ny+2,Nx+2))
    duLy = np.zeros((Ny+2,Nx+2))

    uh = np.zeros((Ny+2,Nx+2))

    # choose limiter type:
    # 0: LF; 1: minmod; 2: MUSCL; 3: Superbee;
    # 4:van Albada; 5:van Leer; 6: TVB
    typ = 1
    c = 0.0
    M = 10.0

    # First step from non-staggered to staggered grid
    # Extend data and assign boundary conditions on global data
    for i in range(Ny):
        xe, ue[i+2,:] = extend(x[0,:], u[i,:], 2, "D", np.pi/4, "D", np.pi/4)

    for j in range(Nx+4):
        ye, ue[:,j] = extend(y[:,0], ue[2:Ny+2,j], 2, "D", np.pi/4, "D", np.pi/4)


    # Compute slopes
    for i in range(Ny+2):
        dup = ue[i+1,2:Nx+4]   - ue[i+1,1:Nx+3]
        dum = ue[i+1,1:Nx+3] - ue[i+1,:Nx+2]
        duLx[i,:] = SlopeLimit( dum, dup, typ, c, M, hx ) 

    for j in range(Nx+2):
        dup = ue[2:Ny+4,j+1]   - ue[1:Ny+3,j+1]
        dum = ue[1:Ny+3,j+1] - ue[:Ny+2,j+1]
        duLy[:,j] = SlopeLimit( dum, dup, typ, c, M, hy )

    # Compute intermediate solution
    uloc = ue[1:Ny+3,1:Nx+3] 
    uh =  uloc - k/4.0*( KPPJac2Dx(uloc)*duLx/hx + \
                       KPPJac2Dy(uloc)*duLy/hy)

    us = np.zeros((Ny-1, Nx-1))

    # Compute flux in x-direction
    for i in range(Ny-1):
        # Left term
        flux1 = 0.25  *( ue[i+2,2:Nx+1] +  ue[i+2, 3:Nx+2]) + \
                0.125 *( duLx[i+1,1:Nx] - duLx[i+1,2:Nx+1]) - \
                k/2.0/hx*( KPPFlux2Dx( uh[i+1,2:Nx+1]) - \
                        KPPFlux2Dx(uh[i+1,1:Nx]))

        # Right term
        flux2 = 0.25  *( ue[i+3,2:Nx+1] +  ue[i+3, 3:Nx+2]) + \
                0.125 *( duLx[i+2,1:Nx] - duLx[i+2,2:Nx+1]) - \
                k/2.0/hx*( KPPFlux2Dx( uh[i+2,2:Nx+1]) - \
                        KPPFlux2Dx(uh[i+2,1:Nx]))

        # Compute rhs
        us[i,:] = 0.5*(flux1 + flux2)

    # Compute flux in y-direction
    for j in range(Nx-1):
        # Down term
        flux1 = 0.25  *( ue[2:Ny+1,j+2] +  ue[3:Ny+2,j+2]) + \
                0.125 *( duLy[1:Ny,j+1] - duLy[2:Ny+1,j+1]) - \
                k/2/hy*( KPPFlux2Dy( uh[2:Ny+1,j+1]) - KPPFlux2Dy(uh[1:Ny,j+1]))

        # Up term
        flux2 = 0.25  *( ue[2:Ny+1,j+3] +  ue[3:Ny+2,j+3]) + \
                0.125 *( duLy[1:Ny,j+2] - duLy[2:Ny+1,j+2]) - \
                k/2/hy*( KPPFlux2Dy( uh[2:Ny+1,j+2]) - KPPFlux2Dy(uh[1:Ny,j+2]))

        # Compute rhs
        us[:,j] += 0.5*(flux1 + flux2)

    # Second step from staggered to non-staggered grid
    Nys = Ny-1
    Nxs = Nx-1

    ue = np.zeros((Nys+4,Nxs+4))

    duLx = np.zeros((Nys+2,Nxs+2))
    duLy = np.zeros((Nys+2,Nxs+2))

    uh = np.zeros((Nys+2,Nxs+2))

    # Extend data and assign boundary conditions on global data
    for i in range(Nys):
        xe, ue[i+2,:] = extendstag(x[0,:Nxs],us[i,:], 2, "D", np.pi/4, "D", np.pi/4)

    for j in range(Nxs+4):
        ye, ue[:,j] = extendstag(y[:Nys,0],ue[2:Nys+2,j], 2, "D", np.pi/4, "D", np.pi/4)

    # Compute slopes
    for i in range(Nys+2):
        dup = ue[i+1,2:Nxs+4]   - ue[i+1,1:Nxs+3]
        dum = ue[i+1,1:Nxs+3] - ue[i+1,:Nxs+2]
        duLx[i,:] = SlopeLimit( dum, dup, typ, c, M, hx ) 

    for j in range(Nxs+2):
        dup = ue[2:Nys+4,j+1]   - ue[1:Nys+3,j+1]
        dum = ue[1:Nys+3,j+1] - ue[:Nys+2,j+1]
        duLy[:,j] = SlopeLimit( dum, dup, typ, c, M, hy )

    # Compute intermediate solution
    uloc = ue[1:Nys+3,1:Nxs+3] 
    uh =  uloc - k/4*( KPPJac2Dx(uloc)*duLx/hx + \
                       KPPJac2Dy(uloc)*duLy/hy) 

    uns = np.zeros((Ny, Nx))

    # Compute flux in x-direction
    for i in range(Nys+1):
        # Left term
        flux1 = 0.25  *( ue[i+1,1:Nxs+2] +  ue[i+1, 2:Nxs+3]) + \
                0.125 *( duLx[i,:Nxs+1] - duLx[i,1:Nxs+2]) - \
                k/2/hx*( KPPFlux2Dx( uh[i,1:Nxs+2]) - \
                        KPPFlux2Dx(uh[i,:Nxs+1]))

        # Right term
        flux2 = 0.25  *( ue[i+2,1:Nxs+2] +  ue[i+2, 2:Nxs+3]) + \
                0.125 *( duLx[i+1,:Nxs+1] - duLx[i+1,1:Nxs+2]) - \
                k/2/hx*( KPPFlux2Dx( uh[i+1,1:Nxs+2]) - \
                        KPPFlux2Dx(uh[i+1,:Nxs+1]))

        # Compute rhs
        uns[i,:] = 0.5*(flux1 + flux2)

    # Compute flux in y-direction
    for j in range(Nxs+1):
        #down term
        flux1 = 0.25  *( ue[1:Nys+2,j+1] +  ue[2:Nys+3,j+1]) + \
                0.125 *( duLy[:Nys+1,j] - duLy[1:Nys+2,j]) - \
                k/2/hy*( KPPFlux2Dy( uh[1:Nys+2,j]) - \
                        KPPFlux2Dy(uh[:Nys+1,j]))

        #up term
        flux2 = 0.25  *( ue[1:Nys+2,j+2] +  ue[2:Nys+3,j+2]) + \
                0.125 *( duLy[:Nys+1,j+1] - duLy[1:Nys+2,j+1]) - \
                k/2/hy*( KPPFlux2Dy( uh[1:Nys+2,j+1]) - \
                        KPPFlux2Dy(uh[:Nys+1,j+1]))

        # Compute rhs
        uns[:,j] += 0.5*(flux1 + flux2)
    # Restore residual

    return (uns-u)/k
