"""File for initialization of Euler equation."""

import numpy as np


def IsentropicVortex2D(x,x0,u0,y,y0,v0,gamma,beta,t):
    """Purpose: compute flow configuration given by
    Y.C. Zhou, G.W. Wei / Journal of Computational Physics 189 (2003) 159 

    """

    Ny,Nx = x.shape
    q = np.zeros((4,Ny,Nx))

    xs = x-u0*t-x0
    ys = y-v0*t-y0

    rr = np.sqrt(xs**2 + ys**2)

    u = u0 - beta*np.exp(1-rr**2)*ys/(2*np.pi)
    v = v0 + beta*np.exp(1-rr**2)*xs/(2*np.pi)

    r = (1 - ((gamma-1)*beta**2*np.exp(2*(1-rr**2))/(16*gamma*np.pi**2)))**(1.0/(gamma-1))
    q[0,:,:] = r
    q[1,:,:] = r*u
    q[2,:,:] = r*v
    q[3,:,:] = r**gamma/(gamma-1)+0.5*r*(u**2+v**2)
    return q

def Riemann2D(x,y,gamma,RiemannProbCase):
    """Purpose: Initialize 2D Riemann problem. 
		Reference: P. Lax and X.-D. Liu, Solution of two-dimensional Riemann problems 
		of gas dynamics by positive schemes, SIAM J. Sci. Comp., 19 (1998), 319?340.
    """
    Ny,Nx = x.shape
    q = np.zeros((4, Ny, Nx))

    if (RiemannProbCase==1):# 4 rarefication waves
        r1= 1.0000
        u1= 0.0000
        v1= 0.0000
        p1= 1.0000

        r2= 0.5197
        u2=-0.7259
        v2= 0.0000
        p2= 0.4000

        r3= 0.1072
        u3=-0.7259
        v3=-1.4045
        p3= 0.0439

        r4= 0.2579
        u4= 0.0000
        v4=-1.4045
        p4= 0.1500

        FinalTime = 0.2
    elif (RiemannProbCase==2): # 4 rarefication waves
        r1= 1.0000
        u1= 0.0000
        v1= 0.0000
        p1= 1.0000

        r2= 0.5197
        u2=-0.7259
        v2= 0.0000
        p2= 0.4000

        r3= 1.0000
        u3=-0.7259
        v3=-0.7259
        p3= 1.0000

        r4= 0.5197
        u4= 0.0000
        v4=-0.7259
        p4= 0.4000

        FinalTime = 0.2
    elif (RiemannProbCase==3): # 4 shocks
        r1= 1.5000
        u1= 0.0000
        v1= 0.0000
        p1= 1.5000

        r2= 0.5323
        u2= 1.2060
        v2= 0.0000
        p2= 0.3000

        r3= 0.1380
        u3= 1.2060
        v3= 1.2060
        p3= 0.0290

        r4= 0.5323
        u4= 0.0000
        v4= 1.2060
        p4= 0.3000

        FinalTime = 0.3
    elif (RiemannProbCase==4): #4 shocks
        r1= 1.1000
        u1= 0.0000
        v1= 0.0000
        p1= 1.1000

        r2= 0.5065
        u2= 0.8939
        v2= 0.0000
        p2= 0.3500

        r3= 1.1000
        u3= 0.8939
        v3= 0.8939
        p3= 1.1000

        r4= 0.5065
        u4= 0.0000
        v4= 0.8939
        p4= 0.3500

        FinalTime = 0.25
    elif (RiemannProbCase==5): # 4 contact waves
        r1= 1.0000
        u1=-0.7500
        v1=-0.5000
        p1= 1.0000

        r2= 2.0000
        u2=-0.7500
        v2= 0.5000
        p2= 1.0000

        r3= 1.0000
        u3= 0.7500
        v3= 0.5000
        p3= 1.0000

        r4= 3.0000
        u4= 0.7500
        v4=-0.5000
        p4= 1.0000

        FinalTime = 0.23
    elif (RiemannProbCase==6): # 4 contact waves
        r1= 1.0000
        u1= 0.7500
        v1=-0.5000
        p1= 1.0000

        r2= 2.0000
        u2= 0.7500
        v2= 0.5000
        p2= 1.0000

        r3= 1.0000
        u3=-0.7500
        v3= 0.5000
        p3= 1.0000

        r4= 3.0000
        u4=-0.7500
        v4=-0.5000
        p4= 1.0000

        FinalTime = 0.3
    elif (RiemannProbCase==7): # 2 contact waves, 2 rarefication waves
        r1= 1.0000
        u1= 0.1000
        v1= 0.1000
        p1= 1.0000

        r2= 0.5197
        u2=-0.6259
        v2= 0.1000
        p2= 0.4000

        r3= 0.8000
        u3= 0.1000
        v3= 0.1000
        p3= 0.4000

        r4= 0.5197
        u4= 0.1000
        v4=-0.6259
        p4= 0.4000

        FinalTime = 0.25
    elif (RiemannProbCase==8): # 2 contact waves, 2 rarefication waves
        r1= 0.5197
        u1= 0.1000
        v1= 0.1000
        p1= 0.4000

        r2= 1.0000
        u2=-0.6259
        v2= 0.1000
        p2= 1.0000

        r3= 0.8000
        u3= 0.1000
        v3= 0.1000
        p3= 1.0000

        r4= 1.0000
        u4= 0.1000
        v4=-0.6259
        p4= 1.0000

        FinalTime = 0.25
    elif (RiemannProbCase==9): # 2 contact waves, 2 rarefication waves
        r1= 1.0000
        u1= 0.0000
        v1= 0.3000
        p1= 1.0000

        r2= 2.0000
        u2= 0.0000
        v2=-0.3000
        p2= 1.0000

        r3= 1.0390
        u3= 0.0000
        v3=-0.8133
        p3= 0.4000

        r4= 0.5197
        u4= 0.0000
        v4=-0.4259
        p4= 0.4000

        FinalTime = 0.3
    elif (RiemannProbCase==10): # 2 contact waves, 2 rarefication waves
        r1= 1.0000
        u1= 0.0000
        v1= 0.4297
        p1= 1.0000

        r2= 0.5000
        u2= 0.0000
        v2= 0.6076
        p2= 1.0000

        r3= 0.2281
        u3= 0.0000
        v3=-0.6076
        p3= 0.3333

        r4= 0.4562
        u4= 0.0000
        v4=-0.4297
        p4= 0.3333

        FinalTime = 0.15
    elif (RiemannProbCase==11): # 2 contact waves, 2 shocks
        r1= 1.0000
        u1= 0.1000
        v1= 0.0000
        p1= 1.0000

        r2= 0.5313
        u2= 0.8276
        v2= 0.0000
        p2= 0.4000

        r3= 0.8000
        u3= 0.1000
        v3= 0.0000
        p3= 0.4000

        r4= 0.5313
        u4= 0.1000
        v4= 0.7276
        p4= 0.4000

        FinalTime = 0.3
    elif (RiemannProbCase==12): # 2 contact waves, 2 shocks
        r1= 0.5313
        u1= 0.0000
        v1= 0.0000
        p1= 0.4000

        r2= 1.0000
        u2= 0.7276
        v2= 0.0000
        p2= 1.0000

        r3= 0.8000
        u3= 0.0000
        v3= 0.0000
        p3= 1.0000

        r4= 1.0000
        u4= 0.0000
        v4= 0.7276
        p4= 1.0000

        FinalTime = 0.25
    elif (RiemannProbCase==13): # 2 contact waves, 2 shocks
        r1= 1.0000
        u1= 0.0000
        v1=-0.3000
        p1= 1.0000

        r2= 2.0000
        u2= 0.0000
        v2= 0.3000
        p2= 1.0000

        r3= 1.0625
        u3= 0.0000
        v3= 0.8145
        p3= 0.4000

        r4= 0.5313
        u4= 0.0000
        v4= 0.4276
        p4= 0.4000

        FinalTime = 0.3
    elif (RiemannProbCase==14): # 2 contact waves, 2 shocks
        r1= 2.0000
        u1= 0.0000
        v1=-0.5606
        p1= 8.0000

        r2= 1.0000
        u2= 0.0000
        v2=-1.2172
        p2= 8.0000

        r3= 0.4736
        u3= 0.0000
        v3= 1.2172
        p3= 2.6667

        r4= 0.9474
        u4= 0.0000
        v4= 1.1606
        p4= 2.6667

        FinalTime = 0.1
    elif (RiemannProbCase==15): # 2 contact waves, 1 shock, 1 rarefication wave
        r1= 1.0000
        u1= 0.1000
        v1=-0.3000
        p1= 1.0000

        r2= 0.5197
        u2=-0.6259
        v2=-0.3000
        p2= 0.4000

        r3= 0.8000
        u3= 0.1000
        v3=-0.3000
        p3= 0.4000

        r4= 0.5313
        u4= 0.1000
        v4= 0.4276
        p4= 0.4000
        FinalTime = 0.2
    elif (RiemannProbCase==16): # 2 contact waves, 1 shock, 1 rarefication wave
        r1= 0.5313
        u1= 0.1000
        v1= 0.1000
        p1= 0.4000

        r2= 1.0222
        u2=-0.6179
        v2= 0.1000
        p2= 1.0000

        r3= 0.8000
        u3= 0.1000
        v3= 0.1000
        p3= 1.0000

        r4= 1.0000
        u4= 0.1000
        v4= 0.8276
        p4= 1.0000

        FinalTime = 0.2
    elif (RiemannProbCase==17): # 2 contact waves, 1 shock, 1 rarefication wave
        r1= 1.0000
        u1= 0.0000
        v1=-0.4000
        p1= 1.0000

        r2= 2.0000
        u2= 0.0000
        v2=-0.3000
        p2= 1.0000

        r3= 1.0625
        u3= 0.0000
        v3= 0.2145
        p3= 0.4000

        r4= 0.5197
        u4= 0.0000
        v4=-1.1259
        p4= 0.4000

        FinalTime = 0.3
    elif (RiemannProbCase==18): # 2 contact waves, 1 shock, 1 rarefication wave
        r1= 1.0000
        u1= 0.0000
        v1= 1.0000
        p1= 1.0000

        r2= 2.0000
        u2= 0.0000
        v2=-0.3000
        p2= 1.0000

        r3= 1.0625
        u3= 0.0000
        v3= 0.2145
        p3= 0.4000

        r4= 0.5197
        u4= 0.0000
        v4= 0.2741
        p4= 0.4000

        FinalTime = 0.2
    elif (RiemannProbCase==19): # 2 contact waves, 1 shock, 1 rarefication wave
        r1= 1.0000
        u1= 0.0000
        v1= 0.3000
        p1= 1.0000

        r2= 2.0000
        u2= 0.0000
        v2=-0.3000
        p2= 1.0000

        r3= 1.0625
        u3= 0.0000
        v3= 0.2145
        p3= 0.4000

        r4= 0.5197
        u4= 0.0000
        v4=-0.4259
        p4= 0.4000

        FinalTime = 0.2
    else: # Incorrect initial conditions
        r1= 0.0000
        u1= 0.0000
        v1= 0.0000
        p1= 0.0000

        r2= 0.0000
        u2= 0.0000
        v2= 0.0000
        p2= 0.0000

        r3= 0.0000
        u3= 0.0000
        v3= 0.0000
        p3= 0.0000

        r4= 0.0000
        u4= 0.0000
        v4= 0.0000
        p4= 0.0000 
        FinalTime = 0.0
        print('Initial condition undefined')

    c1 = (x >= 0.5)*(y >= 0.5)
    c2 = (x < 0.5)*(y >= 0.5)
    c3 = (x < 0.5)*(y < 0.5)
    c4 = (x >= 0.5)*(y < 0.5)

    q[0,:,:] = r1*c1 + r2*c2 + r3*c3 + r4*c4
    q[1,:,:] = r1*u1*c1 + r2*u2*c2 + r3*u3*c3 + r4*u4*c4
    q[2,:,:] = r1*v1*c1 + r2*v2*c2 + r3*v3*c3 + r4*v4*c4
    q[3,:,:] = (p1/(gamma-1)+0.5*r1*(u1**2+v1**2))*c1 + (p2/(gamma-1)+0.5*r2*(u2**2+v2**2))*c2 \
              +(p3/(gamma-1)+0.5*r3*(u3**2+v3**2))*c3 + (p4/(gamma-1)+0.5*r4*(u4**2+v4**2))*c4

    return q, FinalTime
