"""
This file contains the specific functions to solve Euler equations
in 1D or 2D using a ENO scheme.
"""

import numpy as np
from Euler import *
from Eno import *
from helpers import extend

## 1D ##
###################################################################################
def EulerENOrhs1D(x,q,h,k,m,Crec,gamma,maxvel):
    """Purpose: Evaluate the RHS of Euler equations using an ENO reconstruction"""
    N = len(x)
    dq = np.zeros((3,N))

    qe = np.zeros((3,N+2*m))

    # Extend data and assign boundary conditions for Sod's problem
    xe,qe[0,:] = extend(x, q[0,:], m, "D", 1.0, "D", 0.125)
    xe,qe[1,:] = extend(x, q[1,:], m, "D", 0.0, "N", 0.0)
    xe,qe[2,:] = extend(x, q[2,:], m, "D", 2.5, "N", 0.0)

    # # Extend data and assign boundary conditions for Sod's problem
    # xe,qe[0,:] = extend(x, q[0,:], m, "D", 3.857143, "N", 0.0)
    # xe,qe[1,:] = extend(x, q[1,:], m, "D", 10.141852, "D", 0.0)
    # xe,qe[2,:] = extend(x, q[2,:], m, "D", 39.166661, "N", 0.0)


    # define cell left and right interface values
    qm = np.zeros((3,N+2))
    qp = np.zeros((3,N+2))

    for i in range(N+2):
        qm[0,i], qp[0,i] = ENO(xe[i:1+(i+2*(m-1))],qe[0,i:1+(i+2*(m-1))],m,Crec)
        qm[1,i], qp[1,i] = ENO(xe[i:1+(i+2*(m-1))],qe[1,i:1+(i+2*(m-1))],m,Crec)
        qm[2,i], qp[2,i] = ENO(xe[i:1+(i+2*(m-1))],qe[2,i:1+(i+2*(m-1))],m,Crec)

    # Change numerical flux here
    dq = - (EulerLF(qp[:,1:N+1], qm[:,2:N+2], gamma, maxvel) - \
            EulerLF(qp[:,:N], qm[:,1:N+1], gamma, maxvel))/h

    return dq

def EulerENO1D(x,q,h,m,CFL,gamma,FinalTime):
    """ Purpose: Integrate 1D Euler equation until FinalTime using an ENO
        scheme and a 3rd order SSP-RK
    """   
    t = 0.0
    tstep = 0.0
    
    #Initialize reconstruction weights
    Crec = np.zeros((m+1,m))
    for r in range(-1,m):
        Crec[r+1,:] = ReconstructWeights(m,r)


    while t < FinalTime:
        p = (gamma-1)*(q[2,:] - 0.5*(q[1,:]**2)/q[0,:])
        c = np.sqrt(gamma*p/q[0,:])
        maxvel = (c+abs(q[1,:]/q[0,:])).max()
        k = min(FinalTime-t, CFL*h/maxvel)

        #Update solution
        rhsq  = EulerENOrhs1D(x, q,h,k,m,Crec,gamma,maxvel)
        q1 = q + k*rhsq
        rhsq  = EulerENOrhs1D(x,q1,h,k,m,Crec,gamma,maxvel) 
        q2 = (3*q + q1 + k*rhsq)/4
        rhsq  = EulerENOrhs1D(x,q2,h,k,m,Crec,gamma,maxvel) 
        q  = (q + 2*q2 + 2*k*rhsq)/3

        t = t+k
        tstep += 1
        
    return q

