"""Driver script for solving the 1D Euler equations using a DG scheme"""

import numpy as np
import matplotlib.pyplot as plt
from EulerDG import *
from Legendre import LegendreGL

# Order of the method (m), number of elements (N)
m = 1
N = 256

# Set problem parameters
CFL = 0.1
FinalTime = 1.8
gamma = 1.4

# Define domain, materials and initial conditions
xmin = -5.0
xmax = 5.0
L = xmax-xmin

# Generate mesh
VX = xmin + (xmax-xmin)/N*np.arange(N+1)
r,w = LegendreGL(m) 
x = np.outer( np.ones(m+1), VX[:N] ) + 0.5*np.outer(r+1.0, VX[1:N+1] - VX[:N] )
h = (xmax-xmin)/N

# Define domain,materials and initial conditions
q = np.zeros((m+1,3,N))

# Initialize for Sod's problem
# q[:,0,:] = 1.0*(x<0.5) + 0.125*(x>=0.5)
# q[:,2,:] = (1.0*(x<0.5) + 0.1*(x>=0.5))/(gamma-1.0)

# Initialize for shock-entropy problem
q[:,0,:] = 3.857143*(x<-4.0) + (1.0+0.2*np.sin(np.pi*x))*(x>=-4.0)
q[:,1,:] = 3.857143*2.629369*(x<-4.0)
q[:,2,:] = (10.33333*(x<-4.0) + 1.0*(x>=-4.0))/(gamma-1.0) + \
						0.5*q[:,1,:]**2/q[:,0,:]

# Solve Problem
q = EulerDG1D(x,q,h,m,N,CFL,gamma,FinalTime)

# Plot
plt.plot(x[0,:],q[0,0,:])
plt.title("Solution rho at time {}".format(FinalTime))
plt.show()