"""
This file contains the specific functions to solve Euler equations
in 1D using a discontinuous galerkin scheme.
"""

import numpy as np
from Euler import *
from DiscontinuousGalerkin import *
from helpers import extendDG


def EulerDGrhs1D(x,q,h,k,m,N,gamma,S,Ma,VtoE,maxvel):

    """Purpose  : Evaluate the RHS of Euler equations using a DG method"""
    Imat = np.eye(m+1)

    qe = np.zeros((2,3,N+2))

    # Extract solution
    r = q[:,0,:]
    ru = q[:,1,:]
    E = q[:,2,:]

    # Extend data and assign boundary conditions
    # qe[:,0,:] = extendDG( r[VtoE[0],VtoE[1]], "D", 1.0, "D", 0.125 )
    # qe[:,1,:] = extendDG( ru[VtoE[0],VtoE[1]], "D", 0.0, "N", 0.0 )
    # qe[:,2,:] = extendDG( E[VtoE[0],VtoE[1]], "D", 2.5, "N", 0.0 )

    qe[:,0,:] = extendDG( r[VtoE[0],VtoE[1]], "D", 3.857143, "N", 0.0 )
    qe[:,1,:] = extendDG( ru[VtoE[0],VtoE[1]], "D", 10.141852, "D", 0.0 )
    qe[:,2,:] = extendDG( E[VtoE[0],VtoE[1]], "D", 39.166661, "N", 0.0 )

    # Compute volume fluxes
    p = (gamma-1.0)*(E-0.5*ru**2/r)
    fr = ru
    fru = ru**2/r+p
    fE = (E+p)*ru/r

    # Compute surface fluxes
    fluxr = EulerLF( qe[1,:,1:N+1], qe[0,:,2:N+2], gamma, maxvel)
    fluxl = EulerLF( qe[1,:,:N], qe[0,:,1:N+1], gamma, maxvel)

    # Compute residual
    rhsq = np.zeros((2,3,N))

    qh = np.dot(np.transpose(S), fr) - (np.outer(Imat[:,m], fluxr[0,:]) - \
                                        np.outer(Imat[:,0], fluxl[0,:]) )
    
    rhsq[:,0,:] = np.dot( np.linalg.inv(h/2.0*Ma), qh )

    qh = np.dot(np.transpose(S), fru) - (np.outer(Imat[:,m], fluxr[1,:]) - \
                                        np.outer(Imat[:,0], fluxl[1,:]) )
    
    rhsq[:,1,:] = np.dot( np.linalg.inv(h/2.0*Ma), qh )

    qh = np.dot(np.transpose(S), fE) - (np.outer(Imat[:,m], fluxr[2,:]) - \
                                        np.outer(Imat[:,0], fluxl[2,:]) )
    
    rhsq[:,2,:] = np.dot( np.linalg.inv(h/2.0*Ma), qh )

    return rhsq

def EulerDG1D(x,q,h,m,N,CFL,gamma,FinalTime):
    """Purpose  : Integrate 1D Euler equation until FinalTime using a DG
                  scheme and 3rd order SSP-RK method
    """   

    # Initialize operators at Legendre Gauss Lobatto grid
    r,w = LegendreGL(m)
    V = VandermondeDG(m,r)
    D = DmatrixDG(m, r, V)

    Ma = np.linalg.inv(np.dot(V,np.transpose(V)))
    S = np.dot(Ma,D)
    iV = np.linalg.inv(V)

    # Compute operator for WENO smoothness evaluator
    Q,Xm,Xp = WENODGWeights(m,iV)

    # Initialize extraction vector
    VtoE = [np.array(([0]*N, [m]*N)) , np.arange(N)]


    # Compute smallest spatial scale timestep
    rLGLmin = np.abs(r[0]-r[1]).min()

    t = 0.0
    tstep = 0


    #integrate scheme
    while (t<FinalTime):
        # Set timestep
        p = (gamma-1.0)*(q[:,2,:]-0.5*q[:,1,:]**2/q[:,0,:])
        c = np.sqrt(gamma*p/q[:,0,:])
        maxvel = (c+np.abs(q[:,1,:]/q[:,0,:])).max()
        k = min(CFL*rLGLmin*h/maxvel,FinalTime-t)
        
        # Stage 1 of SSPRK
        rhsq  = EulerDGrhs1D(x,q,h,k,m,N,gamma,S,Ma,VtoE,maxvel)
        q1 = q + k*rhsq

        # Limit solution through characteristics
        qc, R = EulerQtoRDG(q1,gamma,V,iV)
        R[:,0,:] = SlopeLimitCSDG(x, R[:,0,:], m, h, N, V, iV)
        R[:,1,:] = SlopeLimitCSDG(x, R[:,1,:], m, h, N, V, iV)
        R[:,2,:] = SlopeLimitCSDG(x, R[:,2,:], m, h, N, V, iV)
        q1 = EulerRtoQDG(R,qc,gamma,V,iV)

        # Update solution - stage 2
        rhsq  = EulerDGrhs1D(x,q1,h,k,m,N,gamma,S,Ma,VtoE,maxvel) 
        q2 = (3*q + q1 + k*rhsq)/4

        # Limit solution through characteristics
        qc, R = EulerQtoRDG(q2,gamma,V,iV)
        R[:,0,:] = SlopeLimitCSDG(x, R[:,0,:], m, h, N, V, iV)
        R[:,1,:] = SlopeLimitCSDG(x, R[:,1,:], m, h, N, V, iV)
        R[:,2,:] = SlopeLimitCSDG(x, R[:,2,:], m, h, N, V, iV)
        q2 = EulerRtoQDG(R,qc,gamma,V,iV)

        # Update solution - stage 3
        rhsq  = EulerDGrhs1D(x,q2,h,k,m,N,gamma,S,Ma,VtoE,maxvel)
        q = (q + 2*q2 + 2*k*rhsq)/3

        # Limit solution through characteristics
        qc, R = EulerQtoRDG(q,gamma,V,iV)
        R[:,0,:] = SlopeLimitCSDG(x, R[:,0,:], m, h, N, V, iV)
        R[:,1,:] = SlopeLimitCSDG(x, R[:,1,:], m, h, N, V, iV)
        R[:,2,:] = SlopeLimitCSDG(x, R[:,2,:], m, h, N, V, iV)
        q = EulerRtoQDG(R,qc,gamma,V,iV)
        
        t += k
        tstep += 1

    return q

def EulerQtoRDG(q,gamma,V,iV):
    """Purpose: Compute characteristic values from conserved values with
                transformation based on cell average in DG formulation."""

    dim = q.shape
    m = dim[0]-1    
    N = dim[2]

    R = np.zeros((m+1,3,N))
    qc = np.zeros((3,N))

    # Extract conserved values and compute cell averages
    r = q[:,0,:]
    mu = q[:,1,:]
    E = q[:,2,:]

    rh = np.dot(iV,r)
    rh[1:m+1,:] = 0.0
    ra = np.dot(V,rh)
    qc[0,:] = ra[0,:]

    muh = np.dot(iV,mu)
    muh[1:m+1,:] = 0.0
    mua = np.dot(V,muh)
    qc[1,:] = mua[0,:]

    Eh = np.dot(iV,E)
    Eh[1:m+1,:] = 0.0
    Ea = np.dot(V,Eh)
    qc[2,:] = Ea[0,:]

    # Compute characteristic values
    for i in range(N):
        S,iS,Lam = EulerChar(qc[:,i],gamma)
        qh = np.transpose(q[:,:,i])
        Ch = np.transpose(np.dot(iS,qh))
        R[:,:,i] = Ch

    return qc,R

def EulerRtoQDG(R,qc,gamma,V,iV):
    """Purpose: Compute conserved values from characteristic values with
                transformation based on cell average in DG formulation."""

    dim = R.shape
    m = dim[0]-1    
    N = dim[2]

    q = np.zeros((m+1,3,N))

    # Compute conserved variables
    for i in range(N):
        S,iS,Lam = EulerChar(qc[:,i],gamma)
        qh = np.transpose(R[:,:,i])
        Ch = np.transpose( np.dot(S,qh) )
        q[:,:,i] = Ch

    return q