"""
This file contains the specific functions to solve Euler equations
in 1D or 2D using a central scheme.
"""

import numpy as np
from Euler import *
from helpers import extend, extendstag, SlopeLimit

## 1D ##
###################################################################################
def EulerCrhs1D(x,q,gamma,h,k,maxvel):
    """Purpose: Evaluate right hand side for Euler equations using second order
            central scheme - NOTE two steps of k/2 is taken
    """ 

    N = len(x)

    duL = np.zeros((3,N+2))
    qh = np.zeros((3,N+2))

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 1
    c = 0.0
    M = 10

    # First step from non-staggered to staggered grid
    qe = np.zeros((3,N+4))

    # Extend data and assign boundary conditions
    xe,qe[0,:] = extend(x,q[0,:], 2, "D", 1.0, "D", 0.125)
    xe,qe[1,:] = extend(x,q[1,:], 2, "D", 0.0, "N", 0.0)
    xe,qe[2,:] = extend(x,q[2,:], 2, "D", 2.5, "N", 0.0)

    # xe,qe[0,:] = extend(x,q[0,:], 2, "D", 3.857143, "N", 0.0)
    # xe,qe[1,:] = extend(x,q[1,:], 2, "D", 10.141852, "D", 0.0)
    # xe,qe[2,:] = extend(x,q[2,:], 2, "D", 39.166661, "N", 0.0)

    # Compute left and right differences, evaluate slopes
    dup = qe[:,2:N+4] - qe[:,1:N+3]
    dum = qe[:,1:N+3] - qe[:,:N+2]
    duL[0,:] = SlopeLimit(dum[0,:],dup[0,:], typ, c, M, h)
    duL[1,:] = SlopeLimit(dum[1,:],dup[1,:], typ, c, M, h)
    duL[2,:] = SlopeLimit(dum[2,:],dup[2,:], typ, c, M, h)

    # Compute intermediate values. f'(u) = Au
    for i in range(N+2):
    	A = EulerJac(qe[:,i+1], gamma)
    	qh[:,i] = qe[:,i+1] - k/(4.0*h)*np.dot(A, duL[:,i])

    # Advance solition k/2
    qs = 0.5*(q[:,:N-1] + q[:,1:N]) + 1.0/8*(duL[:,1:N] - duL[:,2:N+1]) - \
    	 k/2.0/h*(EulerFlux(qh[:,2:N+1], gamma) - EulerFlux(qh[:,1:N], gamma))

    # Second step from staggered to non-staggered grid
    Ns = N-1
    duL = np.zeros((3,Ns+2))
    qe = np.zeros((3,Ns+4))
    xs = x[:Ns]+0.5*h

    # Extend data and assign boundary conditions
    xe,qe[0,:] = extendstag(xs,qs[0,:], 2, "D", 1.0, "D", 0.125)
    xe,qe[1,:] = extendstag(xs,qs[1,:], 2, "D", 0.0, "N", 0.0)
    xe,qe[2,:] = extendstag(xs,qs[2,:], 2, "D", 2.5, "N", 0.0)

    # xe,qe[0,:] = extendstag(xs,qs[0,:], 2, "D", 3.857143, "N", 0.0)
    # xe,qe[1,:] = extendstag(xs,qs[1,:], 2, "D", 10.141852, "D", 0.0)
    # xe,qe[2,:] = extendstag(xs,qs[2,:], 2, "D", 39.166661, "N", 0.0)

    # Compute left and right differences, evaluate slopes
    dup = qe[:,2:Ns+4] - qe[:,1:Ns+3]
    dum = qe[:,1:Ns+3] - qe[:,:Ns+2]

    duL[0,:] = SlopeLimit(dum[0,:],dup[0,:], typ, c, M, h)
    duL[1,:] = SlopeLimit(dum[1,:],dup[1,:], typ, c, M, h)
    duL[2,:] = SlopeLimit(dum[2,:],dup[2,:], typ, c, M, h)
    
    qh = np.zeros((3,Ns+2))

    # Compute intermediate values. f'(u) = Au
    for i in range(Ns+2):
    	A = EulerJac(qe[:,i+1], gamma)
    	qh[:,i] = qe[:,i+1] - k/(4*h)*np.dot(A, duL[:,i])


    # Advance solition k/2
    qns = 0.5*(qe[:,1:Ns+2] + qe[:,2:Ns+3]) + 1.0/8*(duL[:,:Ns+1] - duL[:,1:Ns+2]) - \
    	  k/2.0/h*(EulerFlux(qh[:,1:Ns+2], gamma) - EulerFlux(qh[:,:Ns+1], gamma))
    
    # Restore residual
    dq = (qns-q)/k

    return dq


def EulerC1D(x,q,h,CFL,gamma,FinalTime):
    """Purpose  : Integrate 1D Euler equation until FinalTime using a 
                second order central scheme. 
    """   
    
    t = 0.0
    tstep = 0

    # Integrate scheme
    while t < FinalTime:
        # Set timestep
        p = (gamma-1)*(q[2,:] - 0.5*q[1,:]**2/q[0,:])
        c = np.sqrt(gamma*p/q[0,:])
        maxvel = max(c+np.abs(q[1,:]/q[0,:]))
        k = min(FinalTime-t,CFL*h/maxvel)
        # Update solution
        q += k*EulerCrhs1D(x,q,gamma,h,k,maxvel)

        t += k
        tstep += 1 

    return q



## 2D ##
###################################################################################
def EulerC2D(x,y,q,hx,hy,gamma,CFL,FinalTime):
    """Purpose  : Integrate 2D Euler equation until FinalTime 
                using second order central scheme."""   
    t = 0.0
    tstep = 0

    delta = min(hx,hy)
    while t < FinalTime:
        p = (gamma-1)*(q[3,:,:] - 0.5*(q[1,:,:]**2+q[2,:,:]**2)/q[0,:,:])
        c = np.sqrt(gamma*p/q[0,:,:])
        maxvelx = (c+abs(q[1,:,:]/q[0,:,:])).max()
        maxvely = (c+abs(q[2,:,:]/q[0,:,:])).max()
        k = min(FinalTime-t, CFL*delta/(np.sqrt( 2*(maxvelx**2+maxvely**2) )))
        q += k*EulerC2Drhs(x,y,q,gamma,hx,hy,k)
        t = t+k
        tstep += 1

    return q

def EulerC2Drhs(x,y,q,gamma,hx,hy,k):
    """Purpose: Evaluate right hand side for the two-dimensional Euler equations 
              using a second order central scheme 
    """ 

    Ny,Nx = x.shape

    qe = np.zeros((4,Ny+4,Nx+4))
    qs = np.zeros((4, Ny-1, Nx-1))

    duLx = np.zeros((4,Ny+2,Nx+2))
    duLy = np.zeros((4,Ny+2,Nx+2))

    qh = np.zeros((4,Ny+2,Nx+2))

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 1
    c=0.0
    M=10.0

    # First step from non-staggered to staggered grid
    # Extend data and assign boundary conditions on global data
    for i in range(Ny):
        xe,qe[0,i+2,:] = extend(x[0,:], q[0,i,:], 2, "N", 0, "N", 0)
        xe,qe[1,i+2,:] = extend(x[0,:], q[1,i,:], 2, "N", 0, "N", 0)
        xe,qe[2,i+2,:] = extend(x[0,:], q[2,i,:], 2, "N", 0, "N", 0)
        xe,qe[3,i+2,:] = extend(x[0,:], q[3,i,:], 2, "N", 0, "N", 0)

    for j in range(Nx+4):
        ye,qe[0,:,j] = extend(y[:,1],qe[0,2:Ny+2,j], 2, "N", 0, "N", 0)
        ye,qe[1,:,j] = extend(y[:,1],qe[1,2:Ny+2,j], 2, "N", 0, "N", 0)
        ye,qe[2,:,j] = extend(y[:,1],qe[2,2:Ny+2,j], 2, "N", 0, "N", 0)
        ye,qe[3,:,j] = extend(y[:,1],qe[3,2:Ny+2,j], 2, "N", 0, "N", 0)


    # Compute slopes
    for i in range(Ny+2):
        dup = qe[:,i+1,2:]   - qe[:,i+1,1:-1]
        dum = qe[:,i+1,1:-1] - qe[:,i+1,:-2:]
        duLx[0,i,:] = SlopeLimit( dum[0,:], dup[0,:], typ, c, M, hx ) 
        duLx[1,i,:] = SlopeLimit( dum[1,:], dup[1,:], typ, c, M, hx )
        duLx[2,i,:] = SlopeLimit( dum[2,:], dup[2,:], typ, c, M, hx )
        duLx[3,i,:] = SlopeLimit( dum[3,:], dup[3,:], typ, c, M, hx )

    for j in range(Nx+2):
        dup = qe[:,2:,j+1]   - qe[:,1:-1,j+1]
        dum = qe[:,1:-1,j+1] - qe[:,:-2:,j+1]
        duLy[0,:,j] = SlopeLimit( dum[0,:], dup[0,:], typ, c, M, hy )
        duLy[1,:,j] = SlopeLimit( dum[1,:], dup[1,:], typ, c, M, hy )
        duLy[2,:,j] = SlopeLimit( dum[2,:], dup[2,:], typ, c, M, hy )
        duLy[3,:,j] = SlopeLimit( dum[3,:], dup[3,:], typ, c, M, hy )

    # Compute intermediate solution
    for j in range(Nx+2):
        for i in range(Ny+2):
            Ax = EulerJac2Dx(qe[:,i+1,j+1], gamma)
            Ay = EulerJac2Dy(qe[:,i+1,j+1], gamma)
            qh[:,i,j] = qe[:,i+1,j+1] - k/4*( np.dot(Ax, duLx[:,i,j])/hx + \
                                              np.dot(Ay, duLy[:,i,j])/hy )


    # Compute flux in x-direction
    for i in range(Ny-1):
        # Left term
        flux1 = 0.25  *( qe[:,i+2,2:Nx+1] +  qe[:,i+2, 3:Nx+2]) + \
                0.125 *( duLx[:,i+1,1:Nx] - duLx[:,i+1,2:Nx+1]) - \
                k/2/hx*( EulerFlux2Dx( qh[:,i+1,2:Nx+1] , gamma) - \
                         EulerFlux2Dx(qh[:,i+1,1:Nx] , gamma))

        # Right term
        flux2 = 0.25  *( qe[:,i+3,2:Nx+1] +  qe[:,i+3, 3:Nx+2]) + \
                0.125 *( duLx[:,i+2,1:Nx] - duLx[:,i+2,2:Nx+1]) - \
                k/2/hx*( EulerFlux2Dx( qh[:,i+2,2:Nx+1] , gamma) - \
                         EulerFlux2Dx(qh[:,i+2,1:Nx] , gamma))

        # Compute rhs
        qs[:,i,:] = 0.5*(flux1 + flux2)

    # Compute flux in y-direction
    for j in range(Nx-1):
        # Down term
        flux1 = 0.25  *( qe[:,2:Ny+1,j+2] +  qe[:,3:Ny+2,j+2]) + \
                0.125 *( duLy[:,1:Ny,j+1] - duLy[:,2:Ny+1,j+1]) - \
                k/2/hy*( EulerFlux2Dy( qh[:,2:Ny+1,j+1] , gamma) - EulerFlux2Dy(qh[:,1:Ny,j+1] , gamma))

        # Up term
        flux2 = 0.25  *( qe[:,2:Ny+1,j+3] +  qe[:,3:Ny+2,j+3]) + \
                0.125 *( duLy[:,1:Ny,j+2] - duLy[:,2:Ny+1,j+2]) - \
                k/2/hy*( EulerFlux2Dy( qh[:,2:Ny+1,j+2] , gamma) - EulerFlux2Dy(qh[:,1:Ny,j+2] , gamma))

        # Compute rhs
        qs[:,:,j] += 0.5*(flux1 + flux2)

    # Second step from staggered to non-staggered grid
    Nys = Ny-1
    Nxs = Nx-1


    qe = np.zeros((4,Nys+4,Nxs+4))
    duLx = np.zeros((4,Nys+2,Nxs+2))
    duLy = np.zeros((4,Nys+2,Nxs+2))
    qh = np.zeros((4,Nys+2,Nxs+2))
    qns = np.zeros((4, Ny, Nx))

    # Extend data and assign boundary conditions on global data
    for i in range(Nys):
        xe,qe[0,i+2,:] = extendstag(x[0,:Nxs],qs[0,i,:], 2, "N", 0, "N", 0)
        xe,qe[1,i+2,:] = extendstag(x[0,:Nxs],qs[1,i,:], 2, "N", 0, "N", 0)
        xe,qe[2,i+2,:] = extendstag(x[0,:Nxs],qs[2,i,:], 2, "N", 0, "N", 0)
        xe,qe[3,i+2,:] = extendstag(x[0,:Nxs],qs[3,i,:], 2, "N", 0, "N", 0)

    for j in range(Nxs+4):
        ye,qe[0,:,j] = extendstag(y[:Nys,0],qe[0,2:Nys+2,j], 2, "N", 0, "N", 0)
        ye,qe[1,:,j] = extendstag(y[:Nys,0],qe[1,2:Nys+2,j], 2, "N", 0, "N", 0)
        ye,qe[2,:,j] = extendstag(y[:Nys,0],qe[2,2:Nys+2,j], 2, "N", 0, "N", 0)
        ye,qe[3,:,j] = extendstag(y[:Nys,0],qe[3,2:Nys+2,j], 2, "N", 0, "N", 0)

    # Compute slopes
    for i in range(Nys+2):
        dup = qe[:,i+1,2:]   - qe[:,i+1,1:-1]
        dum = qe[:,i+1,1:-1] - qe[:,i+1,:-2:]
        duLx[0,i,:] = SlopeLimit( dum[0,:], dup[0,:], typ, c, M, hx ) 
        duLx[1,i,:] = SlopeLimit( dum[1,:], dup[1,:], typ, c, M, hx )
        duLx[2,i,:] = SlopeLimit( dum[2,:], dup[2,:], typ, c, M, hx )
        duLx[3,i,:] = SlopeLimit( dum[3,:], dup[3,:], typ, c, M, hx )

    for j in range(Nxs+2):
        dup = qe[:,2:,j+1]   - qe[:,1:-1,j+1]
        dum = qe[:,1:-1,j+1] - qe[:,:-2:,j+1]
        duLy[0,:,j] = SlopeLimit( dum[0,:], dup[0,:], typ, c, M, hy )
        duLy[1,:,j] = SlopeLimit( dum[1,:], dup[1,:], typ, c, M, hy )
        duLy[2,:,j] = SlopeLimit( dum[2,:], dup[2,:], typ, c, M, hy )
        duLy[3,:,j] = SlopeLimit( dum[3,:], dup[3,:], typ, c, M, hy )

    # Compute intermediate solution
    for j in range(Nxs+2):
        for i in range(Nys+2):
            Ax = EulerJac2Dx(qe[:,i+1,j+1], gamma)
            Ay = EulerJac2Dy(qe[:,i+1,j+1], gamma)
            qh[:,i,j] = qe[:,i+1,j+1] - k/4*( np.dot(Ax, duLx[:,i,j])/hx + \
                                              np.dot(Ay, duLy[:,i,j])/hy )


    # Compute flux in x-direction
    for i in range(Nys+1):
        # Left term
        flux1 = 0.25  *( qe[:,i+1,1:Nxs+2] +  qe[:,i+1, 2:Nxs+3]) + \
                0.125 *( duLx[:,i,:Nxs+1] - duLx[:,i,1:Nxs+2]) - \
                k/2/hx*( EulerFlux2Dx( qh[:,i,1:Nxs+2] , gamma) - EulerFlux2Dx(qh[:,i,:Nxs+1] , gamma))

        # Right term
        flux2 = 0.25  *( qe[:,i+2,1:Nxs+2] +  qe[:,i+2, 2:Nxs+3]) + \
                0.125 *( duLx[:,i+1,:Nxs+1] - duLx[:,i+1,1:Nxs+2]) - \
                k/2/hx*( EulerFlux2Dx( qh[:,i+1,1:Nxs+2] , gamma) - EulerFlux2Dx(qh[:,i+1,:Nxs+1] , gamma))

        # Compute rhs
        qns[:,i,:] = 0.5*(flux1 + flux2)

    # Compute flux in y-direction
    for j in range(Nxs+1):
        # Down term
        flux1 = 0.25  *( qe[:,1:Nys+2,j+1] +  qe[:,2:Nys+3,j+1]) + \
                0.125 *( duLy[:,:Nys+1,j] - duLy[:,1:Nys+2,j]) - \
                k/2/hy*( EulerFlux2Dy( qh[:,1:Nys+2,j] , gamma) - \
                        EulerFlux2Dy(qh[:,:Nys+1,j] , gamma))

        # Up term
        flux2 = 0.25  *( qe[:,1:Nys+2,j+2] +  qe[:,2:Nys+3,j+2]) + \
                0.125 *( duLy[:,:Nys+1,j+1] - duLy[:,1:Nys+2,j+1]) - \
                k/2/hy*( EulerFlux2Dy( qh[:,1:Nys+2,j+1] , gamma) - \
                         EulerFlux2Dy(qh[:,:Nys+1,j+1] , gamma))

        # Compute rhs
        qns[:,:,j] += 0.5*(flux1 + flux2)

    # Restore residual 
    return (qns-q)/k
