"""
This file contains the specific functions to solve Burgers equations
in 1D using a spectral approach.
"""

import numpy as np
from Burgers import *
from Spectral import *

## 1D ##
###################################################################################
def BurgersSpecrhs1D(x,u,L):

    """Purpose  : Evaluate the RHS of Burgers equations using spectral Fourier
                  collocation method"""

    # Compute residual - direct form
    du = -2.0*np.pi/L*Fourierdx(u**2,1.0)

    # Compute residual - skew symmetric form
    # du = -2.0*np.pi/L*2.0/3.0*(u*Fourierdx(u,1.0) + Fourierdx(u**2,1.0))

    return du

def BurgersSpec1D(x,u,N,L,CFL,FinalTime):
    """Purpose  : Integrate 1D Burgers equation until FinalTime using a Fourier
                  spectral collocation method and 3rd order SSP-RK method
    """   

    t = 0.0
    tstep = 0
    h = L/(2.0*N+1)

    # Parameter for hyperviscosity
    p = 16.0

    #integrate scheme
    while (t<FinalTime):
        #Decide on timestep
        maxvel = (2*np.abs(u)).max()
        k = min(CFL*h/maxvel,FinalTime-t)
        # Update solution 
        rhsu  = BurgersSpecrhs1D(x,u,L)
        u1 = u + k*rhsu
        u1 = FourierVanishHypVisc(u1,p,k,N,L)
        rhsu  = BurgersSpecrhs1D(x,u1,L)
        u2 = (3*u + u1 + k*rhsu)/4
        u2 = FourierVanishHypVisc(u2,p,k,N,L)
        rhsu  = BurgersSpecrhs1D(x,u2,L)
        u = (u + 2*u2 + 2*k*rhsu)/3
        u = FourierVanishHypVisc(u,p,k,N,L)
        
        t += k
        tstep += 1

    return u
