"""
This file contains the specific functions to solve Burgers equations
in 1D using a ENO scheme.
"""

import numpy as np
from Burgers import *
from Eno import *
from helpers import extend

## 1D ##
###################################################################################
def BurgersENOrhs1D(x,u,h,k,m,Crec,maxvel):

    """Purpose: Evaluate the RHS of Burgers equations using an ENO reconstruction"""
    N = len(x)

    xe,ue = extend(x, u, m, "P", 0, "P", 0) # Periodic boundary conditions
    # xe,ue = extend(x, u, m, "D", 2, "N", 0) # Constant boundary conditions

    #define cell left and right interface values
    um = np.zeros(N+2)
    up = np.zeros(N+2)

    for i in range(N+2):
        um[i], up[i] = ENO(xe[i:1+(i+2*(m-1))],ue[i:1+(i+2*(m-1))],m,Crec)

    # Change numerical flux here
    du = - (BurgersLF(up[1:N+1], um[2:N+2], maxvel) - \
            BurgersLF(up[:N], um[1:N+1], maxvel))/h
    return du

def BurgersENO1D(x,u,h,m,CFL,FinalTime):
    """Purpose: Integrate 1D Burgers equation until FinalTime using a monotone scheme.
    """   
    t = 0.0
    tstep = 0

    # Initialize reconstruction weights
    Crec = np.zeros((m+1,m))
    for r in range(-1,m):
        Crec[r+1,:] = ReconstructWeights(m,r)

    # Integrate scheme
    while (t<FinalTime):
        # Decide on timestep
        maxvel = (2*np.abs(u)).max()
        k = min(CFL*h/maxvel,FinalTime-t)
        # Update solution
        rhsu  = BurgersENOrhs1D(x,u,h,k,m,Crec,maxvel)
        u1 = u + k*rhsu
        rhsu  = BurgersENOrhs1D(x,u1,h,k,m,Crec,maxvel) 
        u2 = (3*u + u1 + k*rhsu)/4
        rhsu  = BurgersENOrhs1D(x,u2,h,k,m,Crec,maxvel)
        u = (u + 2*u2 + 2*k*rhsu)/3
        
        t += k
        tstep += 1

    return u
