"""
This file contains the specific functions to solve Burgers equations
in 1D or 2D using a central scheme.
"""

import numpy as np
from Burgers import *
from helpers import extend, extendstag, SlopeLimit

## 1D ##
###################################################################################
def BurgersCrhs1D(x, u, h, k):
    """Evaluate right hand side for Burgers equation using second order
       central scheme - NOTE two steps of k/2 is taken"""

    N = len(u)
    duL = np.zeros(N+2)
    uh = np.zeros(N+2)
    
    # Chose slope limiter:
    # 0: LF; 1: minmod; 2: MUSCL; 3: Superbee;
    # 4:van Albada; 5:van Leer; 6: TVB
    typ = 2
    c = 0.0
    M = 10.0

    # First step from non-staggered to staggered grid
    
    xe,ue = extend(x, u, 2, "P", 0, "P", 0) #periodic BC
    # xe,ue = extend(x, u, 2, "D", 2.0, "N", 0) # Constant BC

    # Compute element slope and limit
    dup = ue[2:N+4] - ue[1:N+3]
    dum = ue[1:N+3] - ue[:N+2]
    duL = SlopeLimit(dup, dum, typ, c, M, h)

    # Compute limited intermediate solution - f'(u)=2*u
    uh = ue[1:N+3] - BurgersJac(ue[1:N+3])*k/(4.0*h)*duL

    # Advance solution - k/2
    us = 0.5*(u[:N-1] + u[1:N]) + 1.0/8*(duL[1:N] - duL[2:N+1]) - \
    	 k/2.0/h*(BurgersFlux(uh[2:N+1]) - BurgersFlux(uh[1:N]))

    # Second step from staggered to non-staggered grid
    Ns = N-1

    xe,ue = extendstag(x[:Ns]+0.5*h, us, 2, "P", 0, "P", 0) # periodic BC
    # xe,ue = extendstag(x[:Ns]+0.5*h, us, 2, "D", 2.0, "N", 0) # Constant BC

    # Compute element slope and limi
    dup = ue[2:Ns+4] - ue[1:Ns+3]
    dum = ue[1:Ns+3] - ue[:Ns+2]
    duL = SlopeLimit(dum,dup,typ,c,M,h)
    
    # Compute limited intermediate solution - f'(u)=2*u
    uh = ue[1:Ns+3] - BurgersJac(ue[1:Ns+3])*k/(4.0*h)*duL

    # Advance solution k/2
    uns = 0.5*(ue[1:Ns+2] + ue[2:Ns+3]) + 1.0/8*(duL[:Ns+1] - duL[1:Ns+2]) - \
    	  k/2.0/h*(BurgersFlux(uh[1:Ns+2]) - BurgersFlux(uh[:Ns+1]))
    
    du = (uns-u)/k

    return du


def BurgersC1D(x, u, h, CFL, FinalTime):
    """Purpose  : Integrate 1D Burgers equation until FinalTime using a second 
                order central scheme."""
    
    t = 0.0
    tstep = 0

    while t < FinalTime:
        # Decide on timestep
        maxvel = (2*np.abs(u)).max()
        k = min(FinalTime-t,CFL*h/maxvel)
        # Update solution
        u += k*BurgersCrhs1D(x, u, h, k)
        t += k
        tstep += 1

    return u



## 2D ##
###################################################################################
def BurgersC2D(x, y, u, hx, hy, CFL, FinalTime):
    """Purpose  : Integrate 2D Burgers equation until 
                FinalTime using second order central scheme.
    """   
    t = 0.
    delta = min(hx,hy)
    
    # Integrate scheme
    while t < FinalTime:
        # Decide on timestep
        maxvel = (2*np.sqrt(2.0)*np.abs(u)).max()
        k = min(FinalTime-t, CFL*delta/maxvel/2)
        # Update solution
        u += k*BurgersCrhs2D(x,y,u,hx,hy,k,maxvel)
        t = t+k

    return u

def BurgersCrhs2D(x, y, u, hx, hy, k, maxvel):
    """Purpose: Evaluate right hand side for 2D Burgers equation 
                using second order central scheme
    """ 

    Ny,Nx = u.shape

    ue = np.zeros((Ny+4,Nx+4))
    us = np.zeros((Ny-1, Nx-1))
    uns = np.zeros((Ny, Nx))

    duLx = np.zeros((Ny+2,Nx+2))
    duLy = np.zeros((Ny+2,Nx+2))

    uh = np.zeros((Ny+2,Nx+2))

    # Chose slope limiter:
    # 0: LF; 1: minmod; 2: MUSCL; 3: Superbee;
    # 4:van Albada; 5:van Leer; 6: TVB
    typ = 2
    c = 0.0
    M = 10.0

    # First step from non-staggered to staggered grid
    # Extend data and assign boundary conditions on global data
    for i in range(Ny):
        xe, ue[i+2,:] = extend(x[i,:], u[i,:], 2, "P", 0, "P", 0)

    for j in range(Nx+4):
        xe, ue[:,j] = extend(y[:,1], ue[2:Ny+2,j], 2, "P", 0, "P", 0)


    # Compute the slopes
    for i in range(Ny+2):
        dup = ue[i+1,2:]   - ue[i+1,1:-1]
        dum = ue[i+1,1:-1] - ue[i+1,:-2:]
        duLx[i,:] = SlopeLimit( dum, dup, typ, c, M, hx ) 

    for j in range(Nx+2):
        dup = ue[2:,j+1]   - ue[1:-1,j+1]
        dum = ue[1:-1,j+1] - ue[:-2:,j+1]
        duLy[:,j] = SlopeLimit( dum, dup, typ, c, M, hy )

    # Compute intermediate solution
    uloc = ue[1:Ny+3,1:Nx+3]
    uh = uloc - k/4.0* (BurgersJac2Dx(uloc)*duLx/hx + \
                        BurgersJac2Dy(uloc)*duLy/hy)

    # Compute flux in x-direction
    for i in range(Ny-1):
        # Left term
        flux1 = 0.25  *( ue[i+2,2:Nx+1] +  ue[i+2, 3:Nx+2]) + \
                0.125 *( duLx[i+1,1:Nx] - duLx[i+1,2:Nx+1]) - \
                k/2/hx*( BurgersFlux2Dx( uh[i+1,2:Nx+1]) - BurgersFlux2Dx(uh[i+1,1:Nx]))

        # Right term
        flux2 = 0.25  *( ue[i+3,2:Nx+1] +  ue[i+3, 3:Nx+2]) + \
                0.125 *( duLx[i+2,1:Nx] - duLx[i+2,2:Nx+1]) - \
                k/2/hx*( BurgersFlux2Dx( uh[i+2,2:Nx+1]) - BurgersFlux2Dx(uh[i+2,1:Nx]))

        # Compute rhs
        us[i,:] = 0.5*(flux1 + flux2)

    # Compute flux in y-direction
    for j in range(Nx-1):
        # Down term
        flux1 = 0.25  *( ue[2:Ny+1,j+2] +  ue[3:Ny+2,j+2]) + \
                0.125 *( duLy[1:Ny,j+1] - duLy[2:Ny+1,j+1]) - \
                k/2/hy*( BurgersFlux2Dy( uh[2:Ny+1,j+1]) - BurgersFlux2Dy(uh[1:Ny,j+1]))

        # Up term
        flux2 = 0.25  *( ue[2:Ny+1,j+3] +  ue[3:Ny+2,j+3]) + \
                0.125 *( duLy[1:Ny,j+2] - duLy[2:Ny+1,j+2]) - \
                k/2/hy*( BurgersFlux2Dy( uh[2:Ny+1,j+2]) - BurgersFlux2Dy(uh[1:Ny,j+2]))

        # Compute rhs
        us[:,j] += 0.5*(flux1 + flux2)

    Nys = Ny-1
    Nxs = Nx-1

    # Second step from staggered to non-staggered grid
    ue = np.zeros((Nys+4,Nxs+4))
    duLx = np.zeros((Nys+2,Nxs+2))
    duLy = np.zeros((Nys+2,Nxs+2))
    uh = np.zeros((Nys+2,Nxs+2))

    # Extend data and assign boundary conditions on global data
    for i in range(Nys):
        xe, ue[i+2,:] = extendstag(x[i,:Nxs], us[i,:], 2, "P", 0, "P", 0)

    for j in range(Nxs+4):
        xe, ue[:,j] = extendstag(y[:Nys,0], ue[2:Nys+2,j], 2, "P", 0, "P", 0)

    # Compute the slopes
    for i in range(Nys+2):
        dup = ue[i+1,2:]   - ue[i+1,1:-1]
        dum = ue[i+1,1:-1] - ue[i+1,:-2:]
        duLx[i,:] = SlopeLimit( dum, dup, typ, c, M, hx ) 

    for j in range(Nxs+2):
        dup = ue[2:,j+1]   - ue[1:-1,j+1]
        dum = ue[1:-1,j+1] - ue[:-2:,j+1]
        duLy[:,j] = SlopeLimit( dum, dup, typ, c, M, hy )

    # Compute intermediate solution
    uloc = ue[1:Nys+3,1:Nxs+3]
    uh = uloc - k/4.0* (BurgersJac2Dx(uloc)*duLx/hx + \
                        BurgersJac2Dy(uloc)*duLy/hy)

    # Compute flux in x-direction
    for i in range(Nys+1):
        # Left term
        flux1 = 0.25  *( ue[i+1,1:Nxs+2] +  ue[i+1, 2:Nxs+3]) + \
                0.125 *( duLx[i,:Nxs+1] - duLx[i,1:Nxs+2]) - \
                k/2/hx*( BurgersFlux2Dx( uh[i,1:Nxs+2]) - BurgersFlux2Dx(uh[i,:Nxs+1]))

        # Right term
        flux2 = 0.25  *( ue[i+2,1:Nxs+2] +  ue[i+2, 2:Nxs+3]) + \
                0.125 *( duLx[i+1,:Nxs+1] - duLx[i+1,1:Nxs+2]) - \
                k/2/hx*( BurgersFlux2Dx( uh[i+1,1:Nxs+2]) - BurgersFlux2Dx(uh[i+1,:Nxs+1]))

        # Compute rhs
        uns[i,:] = 0.5*(flux1 + flux2)

    # Compute flux in y-direction
    for j in range(Nxs+1):
        # Down term
        flux1 = 0.25  *( ue[1:Nys+2,j+1] +  ue[2:Nys+3,j+1]) + \
                0.125 *( duLy[:Nys+1,j] - duLy[1:Nys+2,j]) - \
                k/2/hy*( BurgersFlux2Dy( uh[1:Nys+2,j]) - BurgersFlux2Dy(uh[:Nys+1,j]))

        # Up term
        flux2 = 0.25  *( ue[1:Nys+2,j+2] +  ue[2:Nys+3,j+2]) + \
                0.125 *( duLy[:Nys+1,j+1] - duLy[1:Nys+2,j+1]) - \
                k/2/hy*( BurgersFlux2Dy( uh[1:Nys+2,j+1]) - BurgersFlux2Dy(uh[:Nys+1,j+1]))

        # Compute rhs
        uns[:,j] += 0.5*(flux1 + flux2)

    # Restore residual
    return (uns-u)/k
