# -*- coding: utf-8 -*-
"""
ESO / 21.09.2024

Méthodes d'estimation - Exercice 4
Propagation de variance
Code de base pour la détérmination de z (hauter) à partir la distance et 
l'angle d'élévation 

Les commandes du programme principal concernent surtout l'introduction des 
données et l'affichage. Les calculs classiques font l'objet de fonctions, 
notamment celles qui furent établies pour la 3e série d'exercices.

On pourrait faire plus court en utilisant des fonctions plus sophistiquées 
de numpy. Parfois, on peut remplacer une boucle par une commande plus directe:
"/" division élément par élément;
"np.diag" pour extraire la diagonale d'une matrice!
"""

import numpy as np

# données numériques

# 0. erreurs de localisation de la sible 
e_x  = -2.5  # mm 
e_y  =  4.0  # mm 
e_z  = -2.0  # mm

# 1. observations
a_r  = 8.4232    # distance rho [m]
b_t  = 89.457    # angle horiz [gon] , noter que 2*pi = 400 gon 
c_a  = 12.283    # angle d'élévation [gon]

# 2. tolérances
epsi_a = 5.0     # mm à 10 m pour la réfléctivité d'un cible 10% 
epsi_b = 10.0    # mgon 
epsi_c = 10.0    # mgon

# 3. corrélations
# On suppose que les mesures sont indépendantes.
corr_ab = 0
corr_ac = 0
corr_bc = 0

# Le vecteur des observations 
l = [a_r, c_a]   # ici sous forme de vecteur-ligne
n = len(l)

# constantes
g2r = np.pi/200.0     # gon -> rad  /facteur d'echele
r2g = 200.0/np.pi     # rad -> gon  /facteur d'echele

# ****************************************************************************
# b) Linéarisation numérique du modèle fonctionnel
# **************************************************************************** 

# Les fonctions y1, y2 et y3 sont définies à part, dans un sous-programme
# qui pourrait inclure n'importe quel nombre de fonctions. 

# valeurs originales des fonctions

from fz import fz

ybulle = fz(l)              # vecteur-colonne
r = len(ybulle)             # nombre de fonctions

print('\n','Valeurs de fonction f(z) pour les mesures (rho, alpha)','\n',ybulle)

# allocation de mémoire
# Ce n'est pas obligatoire, Python fait grandir une matrice à mesure qu'on
# la remplit. Toutefois, ses éléments ne sont plus forcément contigus et
# ceci peut ralentir l'exécution du programme.

F    = np.zeros((r,n))   # matrice jacobienne analytique 
Fnum = np.zeros((r,n))   # matrice jacobienne numérique

# incréments pour linéariser.
# Avec une calculette, on choisit des valeurs faciles, un peu plus faibles
# que les écarts-types. Python calcule avec beaucoup de chiffres et l'on
# peut réduire les incréments pour se rapprocher de la théorie. Ici les
# erreurs sont du même ordre de grandeur et l'on pourrait utiliser le même
# incrément, par exemple delta = 0.1, pour les 3 variables. Toutefois pour
# stimuler une bonne pratique, mieux vaut individualiser les incréments et
# inclure une "boucle en j" dans le calcul de Fnum. Ici on choisit 1/30 de
# la tolérance, soit env. 1/10 de l'écart-type.

delta = np.divide([epsi_a, epsi_c],30)

# valeurs incrémentées des fonctions
# A tour de rôle, on prend la valeur incrémentée de chaque variable et
# l'on recalcule les fonctions.

for j in range(n):
    # En Python, les instructions d'affectation créent des liaisons entre un objet 
    # et une cible, mais elles ne copient pas les objets. C'est pourquoi il faut
    # réaliser l'opération copy().
    l_inc = l.copy()
    l_inc[j] = l_inc[j] + delta[j]   # incrément d'une variable
    F[:,j] = fz(l_inc) 	         # usage auxiliaire de F

	# Pour chaque variable, on soustrait les valeurs originales des fonctions,
	# et l'on divise par l'incrément.	

    Fnum[:,j] = (F[:,j]-ybulle)/delta[j]
    
print('\n','(b) Linéarisation numérique de Fnum','\n',Fnum)

# **************************************************************************** 
# c) Linéarisation analytique du modèle fonctionnel
# **************************************************************************** 

# Attention! Les dérivées analytiques classiques des fonctions
# trigonométriques supposent que l'incrément soit en radians. De même pour
# le logarithme supposé naturel et non décimal.
# Attention: les unités dans F doivent correspondre par type d'observation
# à celles de epsilon ou # de sigma (via Kll) car on va les multiplier.
# A noter: la construction de F est difficile à mécaniser!

b_tr = b_t * g2r 
c_ar = c_a * g2r  

F[0,:] = [np.sin(c_ar),  a_r*np.cos(c_ar)*g2r  ]

print('\n','(c) Linéarisation analytique de F','\n',F)

# **************************************************************************** 
# d) Erreurs maximales
# **************************************************************************** 

epsi_l = [[epsi_a],[epsi_c]]

epsi_y = abs(Fnum) @ (epsi_l)

print('\n','(d) Erreurs maximales des fonctions','\n',epsi_y)

# **************************************************************************** 
# e) Erreurs maximales - erreurs observées (dans tolérances?)
# **************************************************************************** 

eobs_abs   = [[abs(e_z)]]
#err_abs   = abs(err_obs)
delta_err = epsi_y - eobs_abs

print('\n','(e) Erreurs maximales minus erreurs observées','\n',delta_err)

# **************************************************************************** 
# f) Matrice de covariance des mesures
# **************************************************************************** 

# On pourrait étendre le code donné pour une matrice 2x2, mais c'est une
# bonne occasion de mécaniser la construction de Kll avec la fonction
# "corvec2covmat" développée dans la série précédente. Attention! cette
# fonction demande des corrélations en pourcents.

from corvec2covmat import corvec2covmat 

sigma_l = np.divide(epsi_l,3)

corr_l = 100*[ corr_ac,]   # en pourcents


Kll = corvec2covmat(sigma_l,corr_l)

print('\n','(f) Matrice de covariance des mesures','\n',Kll)

# **************************************************************************** 
# g) Matrice de covariance des fonctions
# **************************************************************************** 

Kyy = Fnum @ Kll @ np.transpose(Fnum)

print('\n','(g) Matrice de covariance des fonctions','\n',Kyy)

# **************************************************************************** 
# h) Ecarts-types des fonctions
# **************************************************************************** 

# routine pour extraire les écarts-types et les corrélations

from covmat2cormat import covmat2cormat 
[sigma_y,Ryy] = covmat2cormat(Kyy)

sigma_y = sigma_y.reshape(-1,1)
print('\n','(h) Ecarts-types des fonctions','\n',sigma_y)

# **************************************************************************** 
# i) Comparaison erreur maximale / sigma
# **************************************************************************** 

quot = sigma_y/epsi_y   # division "élément par élément"

# Pour de tels quotients, 2 chiffres significatifs suffisent.
print('\n','(i) Quotients sigma/erreur_max','\n',np.around(quot, decimals=2))

# **************************************************************************** 
# j) Matrice de corrélation des fonctions
# **************************************************************************** 

# Dans une matrice de corrélation, tous les termes sont compris entre -1 et
# 1. Généralement, 2 chiffres significatifs suffisent amplement.

print('\n','(j) Matrice de corrélation des fonctions','\n',np.around(Ryy, decimals=2))

# **************************************************************************** 
# k) Introduire des corrélations entre les mesures.
# **************************************************************************** 

print('\n','---------- On introduit des corrélations ----------')
print(" ---------- et l'on répète les calculs.   ----------")

corr_ab = 0.0
corr_ac = -0.75
corr_bc = 0.25
corr_l = np.multiply(100,[ corr_ac])

print('\n',' Nouveau vecteur de corrélation des mesures, en pourcents')
print(corr_l)

Kll = corvec2covmat(sigma_l,corr_l)

print('\n','(l) Nouvelle matrice de covariance des mesures')
print(Kll)

# **************************************************************************** 
# m) Répéter le calcul de propagation et l'extraction des éléments d'analyse.
# **************************************************************************** 

Kyy = F @ Kll @ np.transpose(F)

print('\n','(m) Nouvelle matrice de covariance des fonctions')
print(Kyy)

[sigma_y,Ryy] = covmat2cormat(Kyy)
sigma_y = sigma_y.reshape(-1,1)

print('\n','Nouveaux écarts-types des fonctions')
print(sigma_y)

quot = sigma_y/epsi_y   # division "élément par élément"

print('\n','Nouveaux quotients sigma/erreur_max')
print(quot)

print('\n','(n) Nouvelle matrice de corrélation des fonctions')
print(Ryy)