package com.epfl.esl.sportstracker.presentation

import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.hardware.Sensor
import android.hardware.SensorEvent
import android.hardware.SensorEventListener
import android.hardware.SensorManager
import android.os.Build
import android.os.Bundle
import android.util.Log
import android.view.WindowManager
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.imageResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Devices
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.constraintlayout.compose.ConstraintLayout
import androidx.wear.compose.material.Text
import com.epfl.esl.sportstracker.HRDao
import com.epfl.esl.sportstracker.HRDatabase
import com.epfl.esl.sportstracker.HREntity
import com.epfl.esl.sportstracker.R
import com.epfl.esl.sportstracker.presentation.theme.SportsTrackerTheme
import com.google.android.gms.wearable.DataClient
import com.google.android.gms.wearable.DataEvent
import com.google.android.gms.wearable.DataEventBuffer
import com.google.android.gms.wearable.DataMapItem
import com.google.android.gms.wearable.MessageClient
import com.google.android.gms.wearable.MessageEvent
import com.google.android.gms.wearable.PutDataMapRequest
import com.google.android.gms.wearable.PutDataRequest
import com.google.android.gms.wearable.Wearable
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import java.util.Timer
import kotlin.concurrent.timerTask

class MainActivity : ComponentActivity(), SensorEventListener, DataClient.OnDataChangedListener,
    MessageClient.OnMessageReceivedListener {
    private var bitmap by mutableStateOf<Bitmap?>(null)
    private var username by mutableStateOf("")
    private var heartRate by mutableIntStateOf(0)

    private lateinit var mSensorManager: SensorManager
    private lateinit var mHeartRateSensor: Sensor
    lateinit var databaseRoom: HRDao

    private var timer = Timer()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)

        mSensorManager = getSystemService(SENSOR_SERVICE) as SensorManager
        mHeartRateSensor = mSensorManager.getDefaultSensor(Sensor.TYPE_HEART_RATE)!!

        val application = requireNotNull(this).application
        val dataSource = HRDatabase.getInstance(application).heartRateDao
        databaseRoom = dataSource

        setContent {
            SportsTrackerTheme {
                HomeScreen(username, heartRate, bitmap)
            }
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M
            && checkSelfPermission("android.permission.BODY_SENSORS")
            == PackageManager.PERMISSION_DENIED
        ) {
            requestPermissions(arrayOf("android.permission.BODY_SENSORS"), 0)
        }
    }

    private fun sendDataToMobile(heartRate: ArrayList<Int>) {
        val dataClient: DataClient = Wearable.getDataClient(this)
        val putDataReq: PutDataRequest = PutDataMapRequest.create("/heart_rate").run {
            dataMap.putIntegerArrayList("HEART_RATE", heartRate)
            asPutDataRequest()
        }
        dataClient.putDataItem(putDataReq)
    }

    override fun onResume() {
        super.onResume()

        Wearable.getDataClient(this).addListener(this)
        Wearable.getMessageClient(this).addListener(this)
        mSensorManager.registerListener(this, mHeartRateSensor, SensorManager.SENSOR_DELAY_UI)
    }

    override fun onPause() {
        super.onPause()

        Wearable.getDataClient(this).removeListener(this)
        Wearable.getMessageClient(this).removeListener(this)
        mSensorManager.unregisterListener(this)
    }

    override fun onDataChanged(dataEvents: DataEventBuffer) {
        dataEvents.filter { it.type == DataEvent.TYPE_CHANGED && it.dataItem.uri.path == "/userInfo" }
            .forEach { event ->
                val receivedImageBytes: ByteArray? =
                    DataMapItem.fromDataItem(event.dataItem).dataMap.getByteArray("profileImage")

                receivedImageBytes?.let {
                    bitmap = BitmapFactory.decodeByteArray(
                        receivedImageBytes,
                        0,
                        receivedImageBytes.size
                    )
                }

                username =
                    DataMapItem.fromDataItem(event.dataItem).dataMap.getString("username") ?: ""
            }
    }

    override fun onSensorChanged(event: SensorEvent?) {
        event?.let {
            heartRate = it.values.get(0).toInt()
        }
    }

    override fun onAccuracyChanged(sensor: Sensor?, accuracy: Int) {}

    override fun onMessageReceived(messageEvent: MessageEvent) {
        if (messageEvent.path == "/command") {
            val receivedCommand: String = String(messageEvent.data)
            if (receivedCommand == "Start") {
                timer = Timer()
                timer.schedule(timerTask {
                    insertHRToRoom(heartRate)
                }, 0, 500)
            } else if (receivedCommand == "Stop") {
                timer.cancel()
            }
        }
    }

    private fun insertHRToRoom(heartRate: Int) {
        var asyncHRToSend: List<HREntity>?
        GlobalScope.launch {
            val valHR = HREntity(HRValue = heartRate)
            databaseRoom.insert(valHR)

            if (databaseRoom.size() >= 5) {
                asyncHRToSend = databaseRoom.getAllHRValues()
                Log.i("MainActivity", "database size $asyncHRToSend")
                val HRListToSend = ArrayList(asyncHRToSend!!.map { hrEntity ->
                    hrEntity.HRValue
                })
                sendDataToMobile(HRListToSend)
                databaseRoom.clear()
            }
        }
    }
}

@Composable
fun HomeScreen(
    username: String,
    heartRate: Int,
    bitmap: Bitmap?,
    modifier: Modifier = Modifier
) {
    ConstraintLayout(modifier = modifier.fillMaxSize()) {
        val (image, usernameConstraint, sensor) = createRefs()

        val context = LocalContext.current

        val displayBitmap = bitmap?.asImageBitmap()
            ?: ImageBitmap.Companion.imageResource(context.resources, R.drawable.ic_logo)

        Image(
            bitmap = displayBitmap,
            contentDescription = stringResource(R.string.logo_sports_tracker),
            modifier = modifier
                .size(100.dp)
                .constrainAs(image) {
                    top.linkTo(parent.top)
                    start.linkTo(parent.start)
                    end.linkTo(parent.end)
                    bottom.linkTo(usernameConstraint.top)
                }
        )
        Text(
            text = username,
            style = TextStyle(fontSize = 24.sp),
            textAlign = TextAlign.Center,
            modifier = modifier
                .constrainAs(usernameConstraint) {
                    top.linkTo(image.bottom)
                    start.linkTo(parent.start)
                    end.linkTo(parent.end)
                    bottom.linkTo(sensor.top)
                }
                .padding(start = 16.dp, end = 16.dp)
        )
        Text(text = "HR = $heartRate",
            modifier = modifier
                .constrainAs(sensor) {
                    top.linkTo(usernameConstraint.bottom)
                    start.linkTo(parent.start)
                    end.linkTo(parent.end)
                    bottom.linkTo(parent.bottom)
                }
                .padding(start = 16.dp, end = 16.dp))
    }
}

@Preview(device = Devices.WEAR_OS_SMALL_ROUND, showSystemUi = true)
@Composable
fun HomeScreenPreview() {
    HomeScreen("Test", 45, null)
}