package com.epfl.esl.sportstracker

import android.annotation.SuppressLint
import android.content.Context
import android.location.Address
import android.location.Geocoder
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import co.yml.charts.common.model.Point
import com.google.android.gms.location.LocationServices
import com.google.android.gms.wearable.DataClient
import com.google.android.gms.wearable.DataEvent
import com.google.android.gms.wearable.DataEventBuffer
import com.google.android.gms.wearable.DataMapItem
import java.io.IOException
import java.util.Locale

data class LocationData(val latitude: Double, val longitude: Double, val description: String)

class ExerciseLiveViewModel : ViewModel(), DataClient.OnDataChangedListener {
    val TAG = ExerciseLiveViewModel::class.java.simpleName
    private val _heartRate = MutableLiveData<Int>(0)
    private val _heartRateList = ArrayList<Point>()
    private val _heartRateListLiveData = MutableLiveData<List<Point>>()
    private val _locationData = MutableLiveData<LocationData>(
        LocationData(
            46.5197,
            6.6323,
            "Lausanne"
        )
    ) //Lausanne as a initial location

    val heartRate: LiveData<Int>
        get() = _heartRate
    val heartRateList: LiveData<List<Point>>
        get() = _heartRateListLiveData
    val locationData: LiveData<LocationData>
        get() = _locationData

    override fun onDataChanged(dataEvents: DataEventBuffer) {
        dataEvents
            .filter { it.type == DataEvent.TYPE_CHANGED && it.dataItem.uri.path == "/heart_rate" }
            .forEach { event ->
                val newValue =
                    DataMapItem.fromDataItem(event.dataItem).dataMap.getIntegerArrayList("HEART_RATE")

                if (!newValue.isNullOrEmpty()) {
                    var nextIndex = _heartRateList.size
                    newValue.forEach { i ->
                        _heartRateList.add(Point(nextIndex.toFloat(), i.toFloat()))
                        nextIndex += 1
                    }
                    val lastElementIndex = _heartRateList.lastIndex
                    _heartRate.value = _heartRateList.get(lastElementIndex).y.toInt()
                    _heartRateListLiveData.value = _heartRateList
                }
            }
    }

    @SuppressLint("MissingPermission")
    fun getLastLocation(context: Context) {
        val locationProvider = LocationServices.getFusedLocationProviderClient(context)
        locationProvider.lastLocation.addOnCompleteListener { task ->
            if (task.isSuccessful && task.result != null) {
                val location = task.result
                val address = getAddress(context, location.latitude, location.longitude)
                val locationData = LocationData(location.latitude, location.longitude, address)
                _locationData.postValue(locationData)
            }
        }
    }

    private fun getAddress(context: Context, latitude: Double, longitude: Double): String {
        var address = ""
        val gcd = Geocoder(context, Locale.getDefault())
        val addresses: List<Address>
        try {
            addresses = gcd.getFromLocation(
                latitude,
                longitude,
                1
            ) as List<Address>
            if (addresses.isNotEmpty()) {
                address = addresses[0].getAddressLine(0)
            }
        } catch (e: IOException) {
            Log.e(TAG, "${e.message}")
        }

        return address
    }
}