package com.epfl.esl.sportstracker

import android.content.Context
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.LifecycleResumeEffect
import androidx.lifecycle.viewmodel.compose.viewModel
import co.yml.charts.axis.AxisData
import co.yml.charts.ui.linechart.LineChart
import co.yml.charts.ui.linechart.model.GridLines
import co.yml.charts.ui.linechart.model.IntersectionPoint
import co.yml.charts.ui.linechart.model.Line
import co.yml.charts.ui.linechart.model.LineChartData
import co.yml.charts.ui.linechart.model.LinePlotData
import co.yml.charts.ui.linechart.model.LineStyle
import co.yml.charts.ui.linechart.model.SelectionHighlightPoint
import co.yml.charts.ui.linechart.model.SelectionHighlightPopUp
import co.yml.charts.ui.linechart.model.ShadowUnderLine
import com.epfl.esl.sportstracker.ui.theme.SportsTrackerTheme
import com.google.android.gms.tasks.Tasks
import com.google.android.gms.wearable.DataClient
import com.google.android.gms.wearable.MessageClient
import com.google.android.gms.wearable.Wearable

@Composable
fun ExerciseLiveScreen(
    dataClient: DataClient,
    modifier: Modifier = Modifier,
    exerciseLiveViewModel: ExerciseLiveViewModel = viewModel()
) {
    val heartRate by exerciseLiveViewModel.heartRate.observeAsState(initial = 0)
    val pointsData by exerciseLiveViewModel.heartRateList.observeAsState(initial = listOf())
    val context = LocalContext.current

    LifecycleResumeEffect {
        dataClient.addListener(exerciseLiveViewModel)
        sendCommandToWear("Start", context)

        onPauseOrDispose {
            dataClient.removeListener(exerciseLiveViewModel)
            sendCommandToWear("Stop", context)
        }
    }

    Column(
        modifier = modifier.fillMaxSize(),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Text(text = stringResource(R.string.heart_rate, heartRate))

        val max = pointsData.maxOfOrNull { it.y }?.toInt() ?: 0
        val min = pointsData.minOfOrNull { it.y }?.toInt() ?: 0
        val steps = pointsData.size - 1

        if (pointsData.isNotEmpty()) {
            val xAxisData = AxisData.Builder()
                .axisStepSize(100.dp)
                .backgroundColor(Color.Blue)
                .steps(steps)
                .axisStepSize(20.dp)
                .labelData { i ->
                    i.toString()
                }
                .labelAndAxisLinePadding(15.dp)
                .build()

            val yAxisData = AxisData.Builder()
                .steps(max - min)
                .backgroundColor(Color.Red)
                .labelAndAxisLinePadding(20.dp)
                .labelData { i ->
                    (i + min).toString()
                }.build()

            val lineChartData = LineChartData(
                linePlotData = LinePlotData(
                    lines = listOf(
                        Line(
                            dataPoints = pointsData,
                            LineStyle(),
                            IntersectionPoint(),
                            SelectionHighlightPoint(),
                            ShadowUnderLine(),
                            SelectionHighlightPopUp()
                        )
                    ),
                ),
                xAxisData = xAxisData,
                yAxisData = yAxisData,
                gridLines = GridLines(),
                backgroundColor = Color.White
            )

            LineChart(
                modifier = Modifier
                    .fillMaxWidth()
                    .height(300.dp),
                lineChartData = lineChartData
            )
        }
    }
}

fun sendCommandToWear(command: String, context: Context) {
    Thread(Runnable {
        val connectedNodes: List<String> = Tasks
            .await(
                Wearable
                    .getNodeClient(context).connectedNodes
            )
            .map { it.id }
        connectedNodes.forEach {
            val messageClient: MessageClient = Wearable
                .getMessageClient(context)
            messageClient.sendMessage(it, "/command", command.toByteArray())
        }
    }).start()
}

@Preview
@Composable
fun ExerciseLivePreview() {
    SportsTrackerTheme {
        val context = LocalContext.current
        val dataClient = Wearable.getDataClient(context)
        ExerciseLiveScreen(dataClient)
    }
}