package com.epfl.esl.sportstracker.presentation

import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.imageResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Devices
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.constraintlayout.compose.ConstraintLayout
import androidx.wear.compose.material.Text
import com.epfl.esl.sportstracker.R
import com.epfl.esl.sportstracker.presentation.theme.SportsTrackerTheme
import com.google.android.gms.wearable.DataClient
import com.google.android.gms.wearable.DataEvent
import com.google.android.gms.wearable.DataEventBuffer
import com.google.android.gms.wearable.DataMapItem
import com.google.android.gms.wearable.Wearable

class MainActivity : ComponentActivity(), DataClient.OnDataChangedListener {
    private var bitmap by mutableStateOf<Bitmap?>(null)
    private var username by mutableStateOf("")

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContent {
            SportsTrackerTheme {
                HomeScreen(username, bitmap)
            }
        }
    }

    override fun onResume() {
        super.onResume()

        Wearable.getDataClient(this).addListener(this)
    }

    override fun onPause() {
        super.onPause()

        Wearable.getDataClient(this).removeListener(this)
    }

    override fun onDataChanged(dataEvents: DataEventBuffer) {
        dataEvents.filter { it.type == DataEvent.TYPE_CHANGED && it.dataItem.uri.path == "/userInfo" }
            .forEach { event ->
                val receivedImageBytes: ByteArray? =
                    DataMapItem.fromDataItem(event.dataItem).dataMap.getByteArray("profileImage")

                receivedImageBytes?.let {
                    bitmap = BitmapFactory.decodeByteArray(
                        receivedImageBytes,
                        0,
                        receivedImageBytes.size
                    )
                }

                username =
                    DataMapItem.fromDataItem(event.dataItem).dataMap.getString("username") ?: ""
            }
    }
}

@Composable
fun HomeScreen(username: String, bitmap: Bitmap?, modifier: Modifier = Modifier) {
    ConstraintLayout(modifier = modifier.fillMaxSize()) {
        val (image, text) = createRefs()

        val context = LocalContext.current

        val displayBitmap = bitmap?.asImageBitmap()
            ?: ImageBitmap.Companion.imageResource(context.resources, R.drawable.ic_logo)

        Image(
            bitmap = displayBitmap,
            contentDescription = stringResource(R.string.logo_sports_tracker),
            modifier = modifier
                .size(100.dp)
                .constrainAs(image) {
                    top.linkTo(parent.top)
                    start.linkTo(parent.start)
                    end.linkTo(parent.end)
                    bottom.linkTo(text.top)
                }
        )
        Text(
            text = username,
            style = TextStyle(fontSize = 24.sp),
            textAlign = TextAlign.Center,
            modifier = modifier
                .constrainAs(text) {
                    top.linkTo(image.bottom)
                    start.linkTo(parent.start)
                    end.linkTo(parent.end)
                    bottom.linkTo(parent.bottom)
                }
                .padding(start = 16.dp, end = 16.dp)
        )
    }
}

@Preview(device = Devices.WEAR_OS_SMALL_ROUND, showSystemUi = true)
@Composable
fun HomeScreenPreview() {
    HomeScreen("Test", null)
}