# This includes all the functions used for the EKV MOSFET model
# Redefine the various functions
import numpy as np
from numpy import log as ln
from numpy import log10 as log
from numpy import sqrt as sqrt
from numpy import exp as exp
from numpy import arctan as atan
from numpy import pi as pi

#--------------------------------------------------
# Large-signal normalized functions
#--------------------------------------------------

# Normalized drain current versus charge (long-channel)
def i_q_long(q):
    return q*(q+1)

# Normalized drain current versus charge (including velocity saturation)
def i_q(qs,lambdac):
    return (4*qs*(qs+1))/(2+lambdac+sqrt((lambdac*(2*qs+1))**2+4*(1+lambdac)))

# Normalized charge versus current (long-channel)
def q_i_long(ic):
    return (sqrt(4*ic+1)-1)/2

# Normalized charge versus current (including velocity saturation)
def q_i(ic,lambdac):
    return (sqrt((lambdac*ic+1)**2+4*ic)-1)/2

# Normalized saturation voltage versus normalized charge
def v_q(q):
    return 2*q+ln(q)

# Inverse function giving q(v) corresponding to the inverse function of v(q) using the Lambert W-function
from scipy.special import lambertw
def q_v_lambert(v):
    return np.real(lambertw(2*exp(v))/2)

# Inverse function giving q(v) corresponding to the inverse function of v(q) using the EKV 2.6 approximation
def q_v(v):
    if v <= -15:
        q0=1/(2+exp(v))
    else:
        if v <-0.35:
            z0=1.55+exp(-v)
        else:
            z0=2/(1.3+v-ln(v+1.6))
    z1=(2+z0)/(1+v+ln(z0))
    q0=(1+v+ln(z1))/(2+z1)
    return q0

# Normalized saturation voltage versus inversion coefficient (long-channel)
def vps_ic_long(ic):
    return ln(sqrt(4*ic+1)-1)+sqrt(4*ic+1)-1-ln(2)

# Normalized saturation voltage versus inversion coefficient (including velocity saturation)
def vps_ic(ic,lambdac):
    q=qi(ic,lambdac)
    return 2*q+ln(q)

# Inversion coefficient versus normalized saturation voltage (long-chanel)
def ic_vps_long(vps):
    return i_q_long(q_v(vps))

# Inversion coefficient versus normalized saturation voltage (including velocity saturation)
def ic_vps(vps,lambdac):
    return i_q(q_v(vps),lambdac)

# Inversion coefficient versus normalized saturation voltage (including velocity saturation)
def ic_vps_lambert(vps,lambdac):
    return i_q(q_v_lambert(vps),lambdac)

# Drain-to-source saturation voltage versus inversion coefficient
# New vdssat function to avoid problems in the inverse function (CE: 9.3.2022)
def vdssat_ic(ic):
    vdssatwi=4
    return 2*sqrt(ic+vdssatwi)

# Inversion coefficient versus drain-to-source saturation voltage
# New inverse vdssat function to avoid having negative ic for vdssat<4
def ic_vdssat(vdssat):
    vdssatwi=4
    if vdssat<vdssatwi:
        ic=0
    else:
        ic=(vdssat/2)**2-vdssatwi
    return ic

# Slope factor versus inversion coefficient for vs=0
def n_ic(ic,gammab,psi0):
    vp=vps_ic_long(ic)
    return 1+gammab/(2*sqrt(psi0+vp))

#--------------------------------------------------
# Small-signal normalized functions
#--------------------------------------------------

# Source transconductance versus inversion coefficient (long-channel)
def gms_ic_long(ic):
    return (sqrt(4*ic+1)-1)/2

# Source transconductance versus inversion coefficient (including velocity saturation)
def gms_ic(ic,lambdac):
    lcic1=lambdac*ic+1
    return (sqrt((lcic1)**2+4*ic)-1)/(lambdac*lcic1+2)

# Inversion coefficient versus normalized transconductance (long-channel)
def ic_gms_long(gms):
    return gms*(gms+1)

# Inversion coefficient versus normalized transconductance (including velocity saturation)
def ic_gms(gms,lambdac):
    if lambdac<=0:
        ic=gms*(gms+1)
    else:
        ic=((1-(gms*lambdac)**2)*(2+lambdac)-gms*lambdac**2-sqrt((2+lambdac)**2-(2*gms*lambdac)**2*(1+lambdac)))/(lambdac**2*((gms*lambdac)**2-1))
    return ic

# Normalized Gm/ID function versus inversion coefficient (long-channel)
def gmsid_ic_long(ic):
    return gms_ic_long(ic)/ic

# Normalized Gm/ID function versus inversion coefficient (including velocity saturation)
def gmsid_ic(ic,lambdac):
    return gms_ic(ic,lambdac)/ic

# Inverse normalized Gm/ID function (long-channel)
def ic_gmsid_long(gmsid):
    return (1-gmsid)/gmsid**2

#--------------------------------------------------
# Thermal noise normalized functions
#--------------------------------------------------

# Thermal noise coefficient (long-channel)
def delta_ic_long(ic):
    qs=q_i_long(ic)
    return 2/3*(qs+3/4)/(qs+1)

# Thermal noise coefficient including short-channel effects
def delta_ic(ic,alpha):
    return delta_ic_long(ic)*(1+alpha*ic)

# Thermal noise excess factor (long-channel)
def gamma_ic_long(ic,n):
    return n*delta_ic_long(ic)

# Thermal noise excess factor including short-channel effects
def gamma_ic(ic,gammawi,alpha):
    return gammawi+alpha*ic

# Thermal noise excess factor including short-channel effects
def gamma_ic2(ic,n,alpha):
    return n*delta_ic(ic,alpha)

# Fano suppression factor (long-channel)
def fano_ic(ic):
    return 2*delta_ic(ic)*gmsid_ic(ic)

# Fano suppression factor including short-channel effects
def fano_ic_vs(ic,lambdac,alpha):
    return 2*delta_ic_vs(ic,alpha)*gmsid_ic_vs(ic,lambdac)

def fano_ic_exp(ic,n,lambdac,gammawi,alpha):
    delta=(gammawi+alpha*ic)/n
    return 2*delta*gmsid_ic_vs(ic,lambdac)
