package codebreaking

object WordLists:
  import scala.io.{Source, Codec}
  import java.util.Locale

  def readWords(fname: String): Set[String] =
    Source.fromResource(fname)(Codec.UTF8)
      .getLines.map(_.toLowerCase(Locale.ROOT))
      .filter(_.forall(c => 'a' <= c && c <= 'z')).toSet

  lazy val enNGSL: Set[String] = readWords("en-ngsl.txt")
  lazy val enDebian: Set[String] = readWords("en-debian.txt")
  lazy val enWordle: Set[String] = readWords("en-wordle.txt")
  lazy val enNYT: Set[String] = readWords("en-nyt.txt")

case class TooManyGuesses(maxGuesses: Int) extends Exception(f"At most $maxGuesses allowed")
case class Solution[Word, Score](
    guesses: List[ScoredGuess[Word, Score]],
    result: Option[Word]
)

extension [Word, Score](puzzle: Puzzle[Word, Score])
  def solve(maxGuesses: Int, secret: Word): Solution[Word, Score] =
    def loop(guesses: List[ScoredGuess[Word, Score]]): Solution[Word, Score] =
      if guesses.length > maxGuesses then
        throw TooManyGuesses(maxGuesses)
      val result = puzzle.guess(guesses)
      result match
        case GuessResult.NoSolution =>
          Solution(guesses, None)
        case GuessResult.FoundSolution(word) =>
          Solution(guesses, Some(word))
        case GuessResult.StillGuessing(guess) =>
          val newGuess = ScoredGuess(guess, puzzle.score(guess, secret))
          loop(newGuess :: guesses)
    // println(f"$secret ->")
    val result = loop(List())
    // println(f"  ${result.guesses}")
    result

extension (bc: BullsAndCows.type)
  def score(guess: BullsAndCows.Word, secret: BullsAndCows.Word) =
    BullsAndCows.Game(secret.length).score(guess, secret)

  def solve(maxGuesses: Int, secret: BullsAndCows.Word): Option[BullsAndCows.Word] =
    BullsAndCows.Game(secret.length).solve(maxGuesses, secret).result

extension (mm: MasterMind.type)
  def score(guess: MasterMind.Word, secret: MasterMind.Word) =
    MasterMind.Game(secret.length).score(guess, secret)

  def solve(maxGuesses: Int, secret: MasterMind.Word): Option[MasterMind.Word] =
    MasterMind.Game(secret.length).solve(maxGuesses, secret).result

extension (wordle: Wordle.type)
  def score(guess: Wordle.Word, secret: Wordle.Word) =
    Wordle.Game(secret.length, WordLists.enDebian).score(guess, secret)

  def solve(
      maxGuesses: Int,
      wordList: Set[Wordle.Word],
      secret: Wordle.Word
  ): Option[Wordle.Word] =
    Wordle.Game(secret.length, wordList).solve(maxGuesses, secret).result
