% Fourier transform of the OFDM signal

close all; clear all;

N = 400 % number of carriers
L = 50 % length of the cyclic prefix
T = 1e-4 % seconds
nOFDMblocks = 10000

% Symbol period (same as sampling period)
Ts = T/N;

% Create the necessary bits
nBits = nOFDMblocks * 2*(2+4); % 2 bits for 4QAM + 4 bits for 16QAM. All of it twice and then multiplied with no. of OFDM blocks.
bits = randi(2,1,nBits)-1;

bitsTo4QAM = bits(1:4*nOFDMblocks);
bitsTo16QAM = bits(length(bitsTo4QAM)+1:end);

% Preparing the decimals for the mapping
QAMdecimals4 = my_bi2de(reshape(bitsTo4QAM,[],2));
QAMdecimals16 = my_bi2de(reshape(bitsTo16QAM,[],4));

% Mapping to QAM
map4 = my_qamMap(4);
map16 = my_qamMap(16);

QAMsymbols4 = map4(QAMdecimals4+1);
QAMsymbols16 = map16(QAMdecimals16+1);

% Creating and filling in the OFDM matrix with symbols
OFDMmatrixFdomain = zeros(N,nOFDMblocks);

% Let us figure out the indices. 
% First we compute the index according to math convention (i=0 for f=0).
% Then we add 1 to all indices (Matlab convention).
% To any frequency of the form k/T there is an unique index i = (k mod N). 
% So, the index i to frequency f_i, i=1,2,3,4, is (f_i*T mod N).
% Taking the Matlab convention into consideration, we have that
% f_i corresponds to index (f_i*T mod N)+1.

f = [-1.5e6 -0.5e6 0.5e6 1.5e6]
ind = mod(f*T,N)+1

OFDMmatrixFdomain(ind(1),:) = QAMsymbols16(nOFDMblocks+1:end);
OFDMmatrixFdomain(ind(2),:) = QAMsymbols4(nOFDMblocks+1:end);
OFDMmatrixFdomain(ind(3),:) = QAMsymbols4(1:nOFDMblocks);
OFDMmatrixFdomain(ind(4),:) = QAMsymbols16(1:nOFDMblocks);

% Creating the time domain matrix
OFDMmatrixTdomain = ifft(OFDMmatrixFdomain);

% Add the cyclic prefix
OFDMmatrixTdomainWithCP = [OFDMmatrixTdomain(N-L+1:N,:); OFDMmatrixTdomain];

% Serializing the data
output = OFDMmatrixTdomainWithCP(:);

% Plotting and setting the correct axis
FTransf = abs(fft(output));
figure;
fstep = 1/Ts/length(FTransf);
fline = -1/(2*Ts):fstep:1/(2*Ts)-fstep; 
plot(fline,fftshift(FTransf));
grid on; xlabel('Frequency [Hz]');
ylabel('|v_F(f)|'); title('OFDM signal spectrum')