import numpy as np
import matplotlib.pyplot as plt
import utilMDC
import my_utilPDC

# RECEIVER_SYNC_SCRIPT
# Simulates data transmission using 4-QAM modulation via root-raised-cosine
# pulses and plots the eye diagram and the signal constellation at the
# output of the matched filter. The idea is to illustrate the effect of
# various synchronization algorithms.


# PN Sequence
pn_seq = np.array([-1, 1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1, -1, 1, -1, 1, -1, 1, -1, 1, 1, -1, -1,
    1, 1, -1, -1, -1, 1, -1, -1, -1, 1, -1, 1, 1, -1, 1, -1, -1, 1, 1, 1, -1, -1, 1, -1, -1, -1,
    -1, 1, -1, -1, 1, -1, 1, -1, -1, 1, -1, -1, 1, 1, -1, 1, 1, -1, 1, 1, 1, -1, -1, -1, 1, 1, 1, 1,
    -1, 1, -1, -1, -1, -1, -1, 1, 1, -1, 1, -1, 1, -1, -1, -1, 1, 1, -1, -1, 1, -1, 1, 1, 1, -1,
    1, 1, -1, -1, -1, -1, 1, 1, 1, -1, 1, -1, 1, 1, 1, 1, -1, -1, 1, 1, 1, 1, 1])

# Parameter selection
nsymbols = int(1e4)
M = int(4)  # 4-ary qam
bitsPerSymb = int(np.log2(M))
nbits = int(nsymbols * bitsPerSymb)
beta = 0.22
SPAN = int(8)
T = 1  # s
SPS = int(50)
Fs = SPS/T
SNR_dB = 20  # dB

# create the bits (0/1 values)
bits = np.random.randint(2, size=nbits)

# create the symbols (sol_bi2de, sol_qamMap, sol_encoder)
MaryData = my_utilPDC.sol_bi2de(np.reshape(bits, (nsymbols, bitsPerSymb)))
# encode
map = my_utilPDC.sol_qamMap(M)
symbols = my_utilPDC.sol_encoder(MaryData, map)
Es = np.var(symbols)  # we need this one for the channel

# plot Tx constellation
plt.scatter(symbols.real, symbols.imag, marker='*')
plt.ylabel('Imaginary')
plt.xlabel('Real')
plt.title('Tx constellation')
plt.grid()
plt.show()

# preamble for synchronization
preamble_symbols = np.array([])  # Use this for no synchronization
# preamble_symbols = np.array([1])
# preamble_symbols = pn_seq

# map symbols to samples (rcosdesign + sol_symbols2samples)
h = my_utilPDC.sol_rcosdesign(beta, SPAN, SPS)
# the pulse is already normalized
samples = my_utilPDC.sol_symbols2samples(np.concatenate((preamble_symbols, symbols), axis=None), h, SPS)

# awgn channel with random delay
received, ch_delay = my_utilPDC.channel(samples, SNR_dB, Es, 8*SPS)
# For debugging you can use the second output argument of channel()
# function to know exactly the channel delay:

# correct the sampling time offset (if we have a preamble)
if preamble_symbols.size > 0:
    preamble_signal = my_utilPDC.sol_symbols2samples(preamble_symbols, h, SPS)
    tau_estim = my_utilPDC.my_estimateTau(received, preamble_signal)

    # For debugging only
    print('Channel delay = %s samples, Estimated delay = %s samples\n' % (ch_delay, tau_estim))

    received = received[tau_estim:]
else:
    print('Channel delay = %s samples.\n' % ch_delay)

# generate the sufficient statistics (sol_sufficientStatistics)
suffStat, mfOutput = my_utilPDC.sol_sufficientStatistics(received, h, SPS)

# remove preamble symbols (if any)
suffStat = suffStat[preamble_symbols.size:]
mfOutput = mfOutput[preamble_symbols.size * SPS:]

# decode (sol_decoder)
MaryData_decoded = my_utilPDC.sol_decoder(suffStat, map)

# plot the constellation at the output of the matched filter (after downsampling)
plt.scatter(suffStat.real, suffStat.imag, marker='*')
plt.ylabel('Imaginary')
plt.xlabel('Real')
plt.title('Rx constellation at the output of the matched filter')
plt.grid()
plt.show()

# plot the eye diagram
# plot the eye diagram
my_utilPDC.sol_eyediagram(mfOutput.real, Fs, T)
plt.title('Eye Diagram - real component')
plt.show()
my_utilPDC.sol_eyediagram(mfOutput.imag, Fs, T)
plt.title('Eye Diagram - imag component')
plt.show()

if MaryData.size > MaryData_decoded.size:
    print('Some sent symbols are lost.\n')
else:
    print('%s out of %s symbols are decoded incorrectly.\n' % (np.sum(MaryData != MaryData_decoded[:MaryData.size]), MaryData.size))
    print('We have a symbol-error rate of %g.\n' % (np.sum(MaryData != MaryData_decoded[:MaryData.size])/np.size(MaryData)))
