% RECEIVER_SYNC_SCRIPT 
% Simulates data transmission using 4-QAM modulation via root-raised-cosine
% pulses and plots the eye diagram and the signal constellation at the
% output of the matched filter. The idea is to illustrate the effect of
% various synchronization algorithms.

clear all;
close all;

% PN Sequence
pn_seq = [-1;1;1;1;1;1;1;-1;-1;-1;-1;-1;-1;-1;1;-1;1;-1;1;-1;1;1;-1;-1;...
    1;1;-1;-1;-1;1;-1;-1;-1;1;-1;1;1;-1;1;-1;-1;1;1;1;-1;-1;1;-1;-1;-1;...
    -1;1;-1;-1;1;-1;1;-1;-1;1;-1;-1;1;1;-1;1;1;-1;1;1;1;-1;-1;-1;1;1;1;1;...
    -1;1;-1;-1;-1;-1;-1;1;1;-1;1;-1;1;-1;-1;-1;1;1;-1;-1;1;-1;1;1;1;-1;...
    1;1;-1;-1;-1;-1;1;1;1;-1;1;-1;1;1;1;1;-1;-1;1;1;1;1;1];

% Parameter selection
nsymbols = 1e4; 
M = 4; % 4-ary qam
bitsPerSymb = log2(M);
nbits = nsymbols*bitsPerSymb;
beta = 0.22;
SPAN = 8;
T = 1; % s
SPS = 50;
Fs = SPS/T;
SNR_dB = 20; % dB

% create the bits (0/1 values)
bits=randi(2,1,nbits)-1;

% create the symbols (my_bi2de + my_qamMap + my_encoder)
MaryData = sol_bi2de(reshape(bits,nsymbols,bitsPerSymb));
map = sol_qamMap(M);
% encode
symbols = sol_encoder(MaryData,map);
Es = var(symbols); % we need this one for the channel

% plot Tx constellation
figure();
plot(symbols, '*');
grid on; xlabel('Re'); ylabel('Im'); 
title('Tx constellation');

% preamble for synchronization
preamble_symbols = [];      % Use this for no synchronization
% preamble_symbols = [1];
% preamble_symbols = pn_seq;

% map symbols to samples (rcosdesign + my_symbols2samples)
h = rcosdesign(beta,SPAN,SPS);
% the pulse is already normalized
samples = sol_symbols2samples([preamble_symbols; symbols],h,SPS);

% awgn channel with random delay
received = channel(samples, SNR_dB, Es, 8*SPS);
% For debugging you can use the second output argument of channel()
% function to know exactly the channel delay:
[received, ch_delay] = channel(samples, SNR_dB, Es, 8*SPS);

% correct the sampling time offset (if we have a preamble)
if ~isempty(preamble_symbols)
    preamble_signal = sol_symbols2samples(preamble_symbols, h, SPS);
    tau_estim   = sol_estimateTau(received, preamble_signal);

    % For debugging only 
    fprintf('Channel delay = %d samples, Estimated delay = %d samples\n', ...
        ch_delay, tau_estim);

    received = received(tau_estim+1:end);
else
    fprintf('Channel delay = %d samples.\n', ch_delay);
end

% generate the sufficient statistics (my_sufficientStatistics)
[suffStat, mfOutput] = sol_sufficientStatistics(received,h,SPS);

% remove preamble symbols (if any)
suffStat = suffStat(numel(preamble_symbols) + 1:end);
mfOutput = mfOutput(numel(preamble_symbols)*SPS + 1:end);

% decode (my_decoder)
MaryData_decoded = sol_decoder(suffStat,map);

% plot the constellation at the output of the matched filter (after downsampling)
figure;
plot(suffStat, '*');
grid on; xlabel('Re'); ylabel('Im');
title('Rx constellation at the output of the matched filter');

% plot the eye diagram
figure;
sol_eyediagram(real(mfOutput), Fs, T); title('Eye Diagram - real component');
figure;
sol_eyediagram(imag(mfOutput), Fs, T); title('Eye Diagram - imag component');


if (numel(MaryData) > numel(MaryData_decoded))
    fprintf('Some sent symbols are lost.\n');
else
    fprintf('%d out of %d symbols are decoded incorrectly.\n', ...
        sum(MaryData_decoded(1:numel(MaryData)) ~= MaryData), ...
        numel(MaryData));
    fprintf('We have a symbol-error rate of %g.\n', ...
        sum(MaryData_decoded(1:numel(MaryData)) ~= MaryData) /...
        numel(MaryData));
end
