import numpy as np

"""Summary of configuration variables
    (access each of these as 'gpsc.<name>')
    
    fs              # Sampling frequency [Hz]
    spch            # Samples per chip
    spc             # Samples per code
    spb             # Samples per bit
    spp             # Samples per block
    chpc            # Chips per code
    cpb             # Codes per bit
    bpp             # Bits per block
    
    Ts              # Sampling time [s]
    Tc              # Time of one code [s]
    Tp              # Time of one block [s]
    bps             # Bit rate [bps]
    cr              # Chip rate [chips/s]
    maxdoppler      # Maximum absolute doppler shift [Hz]
    
    bpw             # Bits per word
    bpsf            # Bits per subframe
    preamble        # GPS subframe preamble
    wpsf            # Words per subframe
    
    fc              # GPS carrier frequency L1 [Hz]
    Omega_dot_e     # Earth's rotational rate [rad/s]
    mu_e            # Earth's universal gravitational constant [m^3/s^2]
    pi_gps          # Pi as defined in the GPS standard
    C               # Speed of light [m/s]
    
    datadir         # Directory where data files are kept"""


######################
# SIMULATION OPTIONS #                                                    %
######################

# Put here any configuration parameters necessary for your simulation.

# Bits per block
bpp = 10


#######################
# BASIC GPS CONSTANTS #                                                     %
#######################

# Samples per chip
spch = 4

# Chips per code
chpc = 1023

# Codes per bit
cpb = 20

# Bit rate [bps]
bps = 50

# GPS carrier frequency L1 [Hz]
fc = 154 * 10.23e6

# GPS subframe preamble
preamble = np.array([1, 0, 0, 0, 1, 0, 1, 1])

# Maximum absolute doppler shift [Hz]
maxdoppler = 10e3  #  = 10 kHz

# Earth's universal gravitational constant [m^3/s^2]
mu_e = 3.986005e14

# Earth's rotational rate [rad/s]
Omega_dot_e = 7.2921151467e-5

# Pi as defined in the GPS standard
pi_gps = 3.1415926535898

# Speed of light [m/s]
C = 299792458

#####################
# DERIVED CONSTANTS #
#####################

# Sampling frequency [Hz] // 4 * 1023 * 20 * 50 = 4 092 000 [Hz]
fs = spch * chpc * cpb * bps

# Sampling time [s]
Ts = 1 / fs

# Samples per code
spc = spch * chpc

# Samples per bit
spb = spc * cpb

# Chip rate [chips/s]
cr = chpc * cpb * bps

# Time of one code [s]
Tc = spc * Ts

# Samples per block
spp = bpp * spb

# Time of one block [s]
Tp = spp * Ts

# Bits per word
bpw = 30

# Words per subframe
wpsf = 10

# Bits per subframe
bpsf = bpw * wpsf


#########################
# SIMULATION PARAMETERS #
#########################

# Data file directory
datadir = '../data'
resultsdir = '../data'

# Number of samples per file
spf = 8179460

# used to store the files
store = False  # True
postfix = '-short'  #'-long'
filetype = 'mat'