function [rp, b, f] = solveRangeEquationsVia_fsolve(satPositions, rho_c_values)
% [RP, B, F] = SOLVERANGEEQUATIONSVIA_FSOLVE(SATPOSITIONS, RHO_C_VALUES)
% solves the receiver position equation system, given the satellite
% position matrix SATPOSITIONS and the corrected pseudorange vector
% RHO_C_VALUES. The dimensions of SATPOSITIONS are NUMSAT x 3, where NUMSAT
% is the number of visible and correctly decoded satellites; RHO_C_VALUES
% is a row vector of NUMSAT elements. 
% Output RP is the receiver position.
% Output B is the difference between the range and the pseudorange.
% F is a row vector of NUMSAT elements. It contains norm(SP-RP)-rho_c-b. 
% In an ideal situation, F=0. It means that all equations can be satisfied
% exactly. Satellites for which the corresponding F-value is large should
% be disregarded, provided that we still have at least 4 satellites.

global gpsc;

% Compute number of satellites, raise an error if there are not enough satellites
numsat = size(satPositions, 1);
if (numsat < 4)
    error('rcvrpos:solveRangeEquationsVia_fsolve:NotEnoughSatellites', 'You need at least 4 satellites to compute the receiver position');
end

rec_pos_and_b_0 = zeros(1,4); % initialize
sol = @(rec_pos_and_b) rangeEquations(satPositions, rec_pos_and_b, rho_c_values); % define the system to be solved and for which variables
options = optimoptions('fsolve', 'Algorithm', 'levenberg-marquardt', 'FunctionTolerance', 1e-6, 'Display', 'off');
recPos_and_b = fsolve(sol, rec_pos_and_b_0, options); % solve

rp = recPos_and_b(1:3);
b = recPos_and_b(4);

f=zeros(numsat,1);
for k=1:numsat
f(k)= norm(satPositions(k,:)-rp) -(rho_c_values(k)+b);
end


% Here we define the system of equations to be solved. The equations are
% exactly the range equations from the lecture notes.
function f = rangeEquations(satPositions, rec_pos_and_b, rho_c_values)
    noSats = size(satPositions, 1);
    f = zeros(1,noSats);
    for i = 1:noSats
        f(i) = norm(satPositions(i,:)-rec_pos_and_b(1:3))-(rho_c_values(i)+rec_pos_and_b(4));
    end

end

end % function solveRangeEquationsVia_fsolve()







