import numpy as np
import scipy.io as sio
import os.path
import warnings

import gpsc as gpsc
import utilGpsEphemerides as uge
from utilGpsEphemerides import Ephemeris

def mainProduceEphemerides():
    """MAINPRODUCEEPHEMERIDES Produces the ephemerides and pseudoranges for the visible satellites
        MAINPRODUCEEPHEMERIDES(SATELLITES) loops over the vector of
        integers SATELLITES, and processes the raw bits received from each
        of these visible satellites to extract the ephemeris data and
        compute the pseudorange For each integer in SATELLITES, the raw
        bits are loaded from file bitsNN-long.mat in the 'data' directory,
        where nn is the satellite number.  When called with no arguments,
        the list of visible satellites to process is taken from file
        'foundSat.mat' in the gpsc.datadir directory (this file is generated
        by FINDSATELLITES) If the parity check is passed and the information
        decoded succesfully, results for each satellite are saved in
        <gpsc.datadir>/ephemerisAndPseudorangeNN.mat, and a list of correctly
        decoded satellites is saved in <gpsc.datadir>/correct_sats.mat"""

    gpsc.postfix = '-long'

    found = sio.loadmat(gpsc.datadir + '/foundSat.mat')
    satellites = found['visible_sats'].flatten()

    correct_sats = []  # subset of visible satellites for which the parity check is correct
    idx_firstBitSubframe = []  # idx to the first bit of subframe for each satellite correctly decoded
    tau_firstBitSubframe = []


    # Process each visible satellite in a loop
    indexSat = 0  # Nicolae: add a growing index which increases only for correct_sats
    for sat in satellites:
        print(sat)
        # load the raw bits
        name = f'bits{sat:02d}{gpsc.postfix}.mat'
        bits_filename = os.path.join(gpsc.datadir, name)
        # bits_filename = '{:s}/bits{:02d}{:s}.mat'.format(gpsc.datadir, sat, gpsc.postfix)
        if os.path.exists(bits_filename):
            print(f'Loading {bits_filename} ...')
            aux = sio.loadmat(bits_filename)
            # print('Taus: %d', aux['taus'])
        
            # Nicolae: if the satellite lost sync, there could be less
            # bits than needed (7*300 + 300), so we discard the satellite
            if aux['decodedBits'].size < 2400:
                print()
                warnings.warn(f'main_produce_ephemerides:notEnoughBits. Satellite {sat:02d} has only '
                              f'{aux["decodedBits"].size} bits decoded, skipping\n', RuntimeWarning)
                continue  # go on to process next satellite
        else:
            warnings.warn(f'mainProduceEphemerides:missingFile. File {bits_filename} does not exist, skipping\n',
                              RuntimeWarning)
            continue  # go on to process next satellite
    
        # filename for results
        ephemeris_filename = f'ephemerisAndPseudorange{sat:02d}.mat'

        # Nicolae: look for subframes [1 2 3] and align everything with respect to the first satellite
        # To be changed. We can work with (1800 + 3-4) bits and we will always find 5 aligned subframes. See my notes.
        # For now the code below is kept as the Matlab code from svn (2021 and earlier)
        try:
            subframes, subframesIDs, idx = uge.getSubframes(aux['decodedBits'].flatten())
            # If no error has been raised so far, we can add the current sat to the list of correctly decoded satellites
            correct_sats.append(sat)
        
            if indexSat == 0:
                # for the first sat, get the index where the sequence of subframes 1 2 3 starts
                currentSf = subframesIDs
                sf123IDtmp = np.where(currentSf[1:] == 1)  # (start from 2nd element to account for the case where we just miss subframe 1 for some sats)
                sf123IDFirstSat = sf123IDtmp[0][0] + 1
                idx = idx + gpsc.bpsf * sf123IDFirstSat  # move the index to the start of subframe no. 1
                ephemeris = uge.readEphemeris(subframes[:, sf123IDFirstSat:sf123IDFirstSat + 3])
                # ephemeris.t_tr = ephemeris.t_tr - 6*sf123IDFirstSat; % if the reference idx is set at 1st frame
                print(f'Found first [1 2 3] group at position {sf123IDFirstSat + 1}. '
                      f'Using subframes {currentSf[sf123IDFirstSat:sf123IDFirstSat+3]} from '
                      f'positions {np.array([sf123IDFirstSat+1, sf123IDFirstSat+2, sf123IDFirstSat+3])}.\n\n')
            
            else:
                # for the other ones, align with the first one
                currentSf = subframesIDs
                sf123IDtmp = np.where(currentSf == 1)[0]
                sf123IDdiff = np.abs(sf123IDtmp[0] - sf123IDFirstSat)  # find the closest [1 2 3] sequence to the first sat reference
                indexClosest = np.argmin(sf123IDdiff)
                sf123ID = sf123IDtmp[indexClosest]
                idx = idx + gpsc.bpsf * sf123ID  # move the index to the start of subframe no. 1
                ephemeris = uge.readEphemeris(subframes[:, sf123ID:sf123ID + 3])
                # ephemeris.t_tr = ephemeris.t_tr - 6*(sf123ID-1); % if the reference idx is set at 1st frame
                print(f'Found first [1 2 3] group at position {sf123ID + 1}. '
                      f'Using subframes {currentSf[sf123ID:sf123ID + 3]} from '
                      f'positions {np.array([sf123ID+1, sf123ID+2, sf123ID+3])}.\n\n')

            idx_firstBitSubframe.append(idx)
            tau_firstBitSubframe.append(aux['taus'].flatten()[idx])
            indexSat += 1

            # store (or compare with solution) obtained ephemerides
            filename = os.path.join(gpsc.datadir, ephemeris_filename)
            if gpsc.store:
                ephemeris.save(filename)
                print(f'Saved ephemeris for satellite {sat:02d} in file {filename}\n')
            else:
                solution = Ephemeris()  # initialize
                solution.load(filename)  # load the values/fields from the file into the structure
                # solution = uge.loadEphemeris(filename)
                # sio.loadmat(filename)
                if ephemeris != solution:
                    print(f'Obtained ephemeris does not match the solution for satellite {sat:02d}')
                else:
                    print(f'Correct ephemeris for satellite {sat}\n')

        # Nicolae: If some predefined error, exit gracefully; rethrow other errors
        except uge.NoStartSubframe as e:
            print(e.args)
            warnings.warn(f'Unable to find the start of a subframe, skipping satellite {sat:02d}\n', RuntimeWarning)
            continue
        except uge.NotEnoughSubframes as e:
            print(e.args)
            warnings.warn(f'Not enough subframes, skipping satellite {sat:02d}\n')
            continue
        except uge.ParityCheckFailed as e:
            print(e.args)
            warnings.warn(f'Parity Check Failed, skipping satellite {sat}\n')
            continue
        except:
            print('Unexpected error, will terminate.\n')
            raise

    # check or store list of correctly decoded satellites
    correct_sats = np.array(correct_sats)
    filename = os.path.join(gpsc.datadir, 'correct_sats.mat')
    if gpsc.store:
        sio.savemat(filename, {'correct_sats': correct_sats})
        print(f'Saved list of correctly decoded satellites to {filename}')
    else:
        solution = sio.loadmat(filename)
        if correct_sats.size != solution['correct_sats'].size or np.any(correct_sats - solution['correct_sats']):
            print('Your list of correctly decoded satellites differs from the solution, you may have done an error\n')

    # now compute the pseudoranges
    if len(tau_firstBitSubframe) > 0:
        tau_ref = min(tau_firstBitSubframe)   # This is the receiver time for which the user position will be computed
        # tau_ref = 0 # If we set this to one, we will compute the position when the receiver starts collecting data
    else:
        raise RuntimeError('There are no successfully decoded satellites to produce ephemerides for.')

    for s, sat in enumerate(correct_sats):

        ephemeris_filename = f'ephemerisAndPseudorange{sat:02d}.mat'
        filename = os.path.join(gpsc.datadir, ephemeris_filename)
        aux = sio.loadmat(os.path.join(gpsc.datadir, f'bits{sat:02d}{gpsc.postfix}.mat'))
        pseudorange = uge.computePseudorange(aux['taus'].flatten(), tau_ref, idx_firstBitSubframe[s])

        # check or store solution
        if gpsc.store:
            # appending to the .mat file
            aux = sio.loadmat(filename)
            aux['pseudorange'] = pseudorange
            sio.savemat(filename, aux)
            print(f'Saved pseudorange for satellite {sat:02d} in file {filename}')
        else:
            solution = sio.loadmat(filename)
            if abs(solution['pseudorange'] - pseudorange) > 2:
                print(f'Obtained pseudorange does not match the solution for satellite {sat:02d}')
            else:
                print(f'Correct pseudorange for satellite {sat:02d}')


if __name__ == '__main__':
    mainProduceEphemerides()
