%% RAPTOR Tutorial 1: Simplest possible RAPTOR run with restructured procedure of RAPTOR model building.
% RAPTOR (RApid Plasma Transport simulatOR) is a code that rapidly
% solves the tokamak 1D plasma transport PDEs.
%
% It has been designed to be modular and simple to run from a single
% script, by modifying parameters.

%% RAPTOR workflow
% A typical stand-alone run of RAPTOR consists of the steps: 
%
% # Define environment (add path)
% # load default configuration
% # Customize default configuration
% # Create RAPTOR model and parameters for custom configuration.
% # Define input variables (actuator time trajectories)
% # Prepare initial conditions
% # Run RAPTOR. 
%
%% The simplest possible RAPTOR run
% Before going into details, we will perform all the above steps once in
% their simplest form.
%
% First, add the relevant paths
close all hidden;
run(fullfile(pwd,'..','RAPTOR_path.m')); % add RAPTOR path
%%
% Then, load the default configuration structure. This structure contains
% the default settings from which to configure a RAPTOR simulation
[config] = RAPTOR_config; % load default params
%%
% This loads the default configuration structure, representing a TCV-type
% plasma setup. We could optionally modify this structure here, but for now
% we will use the default structure directly.
% Generate the following structures for these params: model, params, init,
% geometry g, kinetic profiles v.
[model,params,init,g,v,U] = build_RAPTOR_model(config);

%%
% This generates a RAPTOR _model_, _parameter_, and _initialization_ structure.
% as well as a _g_ and _v_ vector, or matrix.
%
% # The model structure contains everything that can be pre-calculated,
% including matrices with spline bases, fixed profiles, and much more.
% This should not be changed manually but only generated from a call of
% |RAPTOR_config|
% # The params structure contains everything that _can_ be changed between
% two runs.
% # The init structure contains information used to define the initial
% condition in the |RAPTOR_initial_conditions| function (see later)
% # g contains information about geometric quantities related to the equilibrium, which may or
% may vary in time.
% # v contains information about those kinetic profiles and other quantities which are not 
% in the state (not solved for) and are pre-assigned,
% and also H-mode and sawtooth crash information.
% # U contains the actuator inputs in time. 

% Define initial conditions for state (depends for example on Ip0 (u(1,1)
% and other inputs
% * x is the state vector, which is updated by solving the PDEs.
% So should be called last before RAPTOR_predictive
x0 = RAPTOR_initial_conditions(model,init,g(:,1),v(:,1)); 

% Define initial conditions for state
%% 
% Now we are ready to run RAPTOR - this will display some plasma quantities as
% they evolve. Change the verbosity level to control the output.
simres = RAPTOR_predictive(x0,g,v,U,model,params,'verbosity',2);

% To compute some physical outputs, we run the RAPTOR_out function.
out = RAPTOR_out(simres,model,params);


%%
% Finally we plot some output quantities
subplot(121); plot(out.time,out.te0); xlabel('t[s]');ylabel('T_{e,0} [eV]');
subplot(122); plot(out.rho,out.q); xlabel('\rho_{tor,N}');ylabel('q [-]');
