function LX = ts2meq(L,LXts,twin)
%   LX = ts2meq(L,LXts,twin)
%
% Transforms structure of timeseries into meq data structure
% Uses meqsize(L,field) to determine sizes of various fields.
% Inputs:
%  L: MEQ structure used for determining field sizes
%  LXts: structure of time series on same time basis, as obtained
%    e.g. from Simulink log or from SCD mems
%  twin: vector with [tstart,tend] for cropping signal (default: take whole
%    signal)
% Outputs:
%  LX: Data structure in MEQ format
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

fields = fieldnames(LXts);
tt = LXts.(fields{1}).Time'; % time of time series
%%

if nargin>2 && ~isempty(twin)
  assert(numel(twin)==2,'twin must have 2 entries')
  it = (tt>=twin(1) & tt<=twin(2));
else
  it = 1:numel(tt);
end

LX.t = tt(it);
nt = numel(LX.t);
for ii=1:numel(fields)
  myfield = fields{ii};
  if isequal(myfield,'t'), continue; end
  tsize = meqsize(L,myfield);
  assert(isequal(LXts.(myfield).Time',tt),'Time base for %s is not equal to that of %s',myfield,fields{1});

  if any(tsize==0),        continue; end % empty fields are not treated correctly in Simulink.
 
  if numel(LXts.(myfield).Data)~=prod([tsize,nt])
    disp('timeseries size:'); disp(size(LXts.(myfield).Data));
    disp('meqsize        :'); disp([tsize,nt]);
    error('size mismatch for %s',myfield);
  end
  
  if LXts.(myfield).IsTimeFirst
    if tsize(2)==1
      LX.(myfield) = reshape(LXts.(myfield).Data(it,:)',tsize(1),nt);
    else
      LX.(myfield) = reshape(LXts.(myfield).Data(it,:)',tsize(1),tsize(2),nt);
    end
  else
    if tsize(2)==1
      LX.(myfield) = reshape(LXts.(myfield).Data(:,:,it),tsize(1),nt);
    else
      LX.(myfield) = reshape(LXts.(myfield).Data(:,:,it),tsize(1),tsize(2),nt);
    end
  end
  
  % specials
  if isequal(myfield,'Ie')
    LX.Ia = LX.Ie(1:L.G.na      ,:);
    LX.Iu = LX.Ie((L.G.na+1):end,:);
    LX.Is = L.G.Tius*LX.Iu;
    LX = rmfield(LX,'Ie');
  end
  
  % auxiliaries
  if isfield(L.P,'shot')
    LX.shot = L.P.shot;
  end
end