classdef csdec_test < genlib_test
  % Test for csdec (cubic spline decomposition)
  %
  % [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    T_ = linspace(0,1,21);
    x_ = linspace(0,1,21);
    tol_ = 1e-8;
  end
  
  properties (TestParameter)
    ec = {'a','n','p','s','t','u','v','w'}
  end

  methods (Test)
    function csdec_ec_test(testCase,ec)
      % csdec_ec_test verify that correct boundary conditions are enforced
      
      T = testCase.T_;
      x = testCase.x_;
      tol = testCase.tol_;

      [M,tau] = csdec(T,x,ec);

      switch ec
        case 'a'
          C = bspsum(tau,M,(T([1:2,end-2:end-1])+T([2:3,end-1:end]))/2,3);
          testCase.verifyEqual(C(1,:),C(2,:),'AbsTol',tol,'Failed continuity condition of third derivative at T(    2) with ec=''a''');
          testCase.verifyEqual(C(3,:),C(4,:),'AbsTol',tol,'Failed continuity condition of third derivative at T(end-1) with ec=''a''');
        case 'n'
          C = bspsum(tau,M,T([1,end]),2);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero second derivative at T(  1) with ec=''n''');
          testCase.verifyLessThan(max(abs(C(2,:))),tol,'Failed check for zero second derivative at T(end) with ec=''n''');
        case 'p'
          C = bspsum(tau,M,T([1,end]),1);
          testCase.verifyEqual(C(1,:),C(2,:),'AbsTol',tol,'Failed periodicity condition of first derivative with ec=''p''');
          C = bspsum(tau,M,T([1,end]),2);
          testCase.verifyEqual(C(1,:),C(2,:),'AbsTol',tol,'Failed periodicity condition of second derivative with ec=''p''');
        case 's'
          C = bspsum(tau,M,T(1),1);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero first derivative at T(  1) with ec=''s''');
          C = bspsum(tau,M,T(end),2);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero second derivative at T(end) with ec=''s''');
        case 't'
          C = bspsum(tau,M,T(1),1);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero first derivative at T(  1) with ec=''t''');
          C = bspsum(tau,M,T(1),3);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero third derivative at T(  1) with ec=''t''');
        case 'u'
          C = bspsum(tau,M,T(1),1);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero first derivative at T(  1) with ec=''u''');
          C = bspsum(tau,M,T(1),3);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero third derivative at T(  1) with ec=''u''');
          C = bspsum(tau,M,T(end),2);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero second derivative at T(end) with ec=''u''');
        case 'v'
          C = bspsum(tau,M,T(1),1);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero first derivative at T(  1) with ec=''v''');
          C = bspsum(tau,M,T(1),3);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero third derivative at T(  1) with ec=''v''');
          C = bspsum(tau,M,T(end),3);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero third derivative at T(end) with ec=''v''');
        case 'w'
          C = bspsum(tau,M,T(1),1);
          testCase.verifyLessThan(max(abs(C(1,:))),tol,'Failed check for zero first derivative at T(  1) with ec=''w''');
          C = bspsum(tau,M,(T(end-2:end-1)+T(end-1:end))/2,3);
          testCase.verifyEqual(C(1,:),C(2,:),'AbsTol',tol,'Failed continuity condition of third derivative at T(end-1) with ec=''w''');
      end
    end
  end

end
