% FGE forward solver time evolving default parameters overloading from sources
% P = fgep(P)
% 
% Many parameters are in common with FGSP.
% FGE-specific parameters:
% .cde         Current Diffusion Equation to be used
% .stoponalarm Stop simulation if coil alarm is triggered
% .nnoc        Number of subsequent non-converging time steps allowed
% .lin         Run linearized version 
% .fixplasma   Fix plasma in place during iterations, no Iy update
% .mey         Scales Mey - set to 0 to avoid back-coupling between plasma and vessel currents (for testing).
% .implicit    Solve implicitly or explicit in Ie,Iy
% .ssinit      Initialize to a steady-state condition (possibly modifying coil/vessel currents)
% .ohcoils     Names of coils used for driving Ohmic current (from G.dima)
% .compcoils   Names of coils used for compensation of stray fields from Ohmic transformer coils
% .wcompa      Weight of regularization term to keep compensation currents small
% .wioha       Weight of regularization term to keep Ohmic current driving currents small
% .ohequal     Force currents of coils for OH induction to be equal
% .compregion  Grid region on which to do compensation ('Iy','Oasx','Oasx-sym', or direct specification)
% .iLpext      Source of Lp (plasma inductance) used for rigid model CDE. true -> provided externally in LX, false -> analytic expression (Iy Myy Iy)/Ip^2 equivalent
% .ctrlfct     Feedback controller function handle (default: [] meaning open-loop simulation)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function [P] = fgep(P,varargin)

D = fgsp(P,varargin{:}); % start with fgs default parameters

%% Default Parameters
% NL solver parameters 
D.mkryl = 200; % krylov solver space. Can be large since GMRES algoritims terminate automatically. 
D.tolF = 1e-8; % default value of tolF (minimum tolerance max(abs(F(x))))
D.usepreconditioner = true; % if active preconditioner
D.relax = 0; % Apply relaxation on solution
% Input source
D.cde = []; % empty for none (no CDE, Ip externally imposed as constraint), otherwise string for handle for cde type
D.lin = false; % run linearized version
D.nnoc = 0; % number of consecutive non-converging time steps allowed
D.fixplasma = 0; % Fix plasma in place during iterations, no Iy update
D.mey = 1; % Scales Mey - set to 0 to avoid back-coupling between plasma and vessel currents (for testing).
D.implicit = true; % solve implicitly or explicit in Ie,Iy

D.ssinit     = true; % initialize to a steady-state condition (possibly modifying coil/vessel currents)
% inductive/compensation coils
D.ohcoils    = ''; % Names of coils used for driving Ohmic current
D.compcoils  = ''; % Names of coils used for compensation of stray fields
D.wcompa     = 1.0e-9; % weight of regularization term to keep compensation currents small
D.wioha      = 0.5e-10; % weight of regularization term to keep Ohmic current driving currents small
D.ohequal    = false; % Force currents of coils for OH induction to be equal
D.compregion = 'Oasx-sym'; % Grid region on which to do compensation ('Iy','Oasx''Oasx-sym', or direct specification)

D.iLpext = true; % Type of used for rigid model CDE. true -> provided externally in LX, false -> analytic expression (Iy Myy Iy)/Ip^2 equivalent

% Initialization source
D.shot = NaN;
D.ctrlfct = []; % Feedback controller function handle (default: open-loop)

D.stoponalarm = true; % stop simulation if coil alarm is triggered

%% Set defaults
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

%% order
P = orderfields(P);

%% Checks
assert(isnumeric(P.mey) && any(P.mey==[0 1]),'mey0 must be 0 or 1');
assert(P.LYall,'LYall must be true for FGE')

end
