function L = fgec(P,G)
% FGEC Parameter consolidation for fge
% L = fgec(P,G)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Generic fgsc ancilliary data (includes meqc)
L = fgsc(P,G);

%% Code specific functions
L.codeinit = @fgeinit; % Initialise the equilibrium based on fgeinit.m
L.codeF = @fgsFlin;    % Call the linearisation for the fgs force balance
L.code = 'fge';

%% additional indices for FGE conductors
L.ind.ia =          L.nN + (1:G.na);
L.ind.iu = L.ind.ia(end) + (1:G.nu);
L.nN = L.nN + G.na + G.nu;  % total number of states

%% Set scaling for residuals
I0 = P.b0/P.r0/(4e-7*pi)/(L.nx);
if strcmp(P.selu, 'e'),   Iu0 = abs(max(max(G.Tvu\(I0*ones(G.nv, 1) ))));
else, Iu0 = I0; end

switch lower(L.P.algoNL)
  case 'all-nl'
    L.resscal(L.ind.ia) = 1./I0;
    L.resscal(L.ind.iu) = 1./Iu0;
    L.xscal = ones(L.nN,1);
  case 'picard'
    % no additional nonlinear states
  otherwise
    error('unknown algoNL=%s',L.P.algoNL)
end

%% Define matrices for temporal evolution problem
Mya = G.Mxa(L.lxy(:),:); % Mutal between internal grid and active coils
Myu = G.Mxu(L.lxy(:),:); % Mutual between internal grid and vessel segments

L.Mey = P.mey*[Mya.' ; Myu.']; % Mutual between external currents and plasma current

L.Mee = [G.Maa , G.Mau; G.Mau.', G.Muu]; % Mutual inductance between external currents
L.Re  = [G.Ra;G.Ru]; % Resistance of external currents
L.iMee = L.Mee\eye(size(L.Mee));

%% Coils for providing inductive voltage or compensation field
assert(~iscell(P.ohcoils)   || ~isempty(P.ohcoils)  ,'  ohcoils can not be an empty cell array')
assert(~iscell(P.compcoils) || ~isempty(P.compcoils),'compcoils can not be an empty cell array')
L.ioha   = contains(G.dima,P.ohcoils);
L.icompa = contains(G.dima,P.compcoils);

end
