# MEQ - Matlab Equilibrium Toolbox

**A suite of codes for tokamak equilibrium calculations.**

*Swiss Plasma Center EPFL Lausanne 2022. All rights reserved*


Originally based on `FBT` and `LIUQE` codes written for the TCV tokamak in `FORTRAN` by F. Hoffmann and others (CRPP-EPFL Lausanne) [1],[2],[3]

Re-written in `MATLAB` by Jean-Marc Moret, including real-time application on the TCV Tokamak [4]

Presently maintained on https://gitlab.epfl.ch/spc/tcv/tbx/meq

Code suite maintainers: 

* [Antoine Merle](mailto:antoine.merle@epfl.ch)
* [Federico Felici](mailto:federico.felici@epfl.ch)

## Installation:

To install for the first time, first ask the maintainers for access to both `meq` and `genlib` toolboxes. 
### Get the code
```
# Clone, e.g.
git clone git@gitlab.epfl.ch:spc/tcv/tbx/meq meq
# initialize and update submodules
git submodule update --init --recursive
```

### Build

`meq` is built using the `make` command. The default `all-nort` target  builds all necessary binaries to successfuly run the test suite including for the Simulink models. It is highly recommended to provide the path to the MATLAB installation using the `MATPATH` variable. Other commonly used variables to customize the build are `CC` to choose the C compiler and `BLASLIB` to choose the source of the BLAS and LAPACK libraries (see below for a more detailed description).

For example the following line builds `meq` with GCC and the MATLAB-provided version of BLAS and LAPACK.
```
# make meq
make all-nort CC=gcc MATPATH=<path_to_matlab>
```
#### General makefile options:

`meq` makefile targets:

* `tbx` (`mex`es, for `.m` use only)
* `doc` (documentation)
* `all` (includes Simulink S functions, rt library, and genlib)
* `all-nort` (same as `all` but excluding the rt library, this is the default target)

`meq`/`genlib` makefile options:

* `CC`: Compiler (`gcc` or `icc`, default is `icc`)
* `CCRT`: Compiler for real-time version
* `MATPATH`: Path of `MATLAB` installation (e.g. `$MATPATH/bin/mex` must exist) 
* `BLASLIB`: Default links to `MATLAB`-supplied MKL. Otherwise specify `MKL` or `CBLAS` (tested with OpenBlas on OSX)
* `BUILD`: `optimized` (default) or `debug` (allows debugging of MEX-files using e.g. `gdb` and prints debug information from `libmeq` routines)



#### On SPC LAC clusters 
On a machine with icc:

```
make all
```

#### On HPC ITER hpc-login.iter.org
```
# Clean existing loaded module 
#   (to avoid conflict with compiler version and matlab version)
module purge
# Load intel compiler 
module load intel/2018a
# Load matlab module
module load MATLAB
# Set environment variable for matlab root
export MATLAB=$EBROOTMATLAB
# Compile (from within meq folder)
make all
```

#### On MacOS on an ARM processor
First check the architecture with `arch`, if it is `arm` then first do:
```
$ env /usr/bin/arch - x86_64 /bin/zsh --login
```
to change the architecture. After this command, `arch` should return `i386`

If you want to use `icc` you probably need to run
```
source /opt/intel/oneapi/setvars.sh 
make CC=icc
```

#### On a Windows machine
The windows build of MEQ is still under development. For now we recommend to have a machine with MATLAB and MSYS2 installed, and within the MSYS2-MINGW64 shell install the `make` and `git` packages as well as the `mingw-w64-x86_64-toolchain` group using the `pacman -S make git mingw-w64-x86_64-toolchain` command.

Within a MSYS2-MINGW64 shell, run the following command from the `meq` directory:
```
make MATPATH="/c/Program\ Files/MATLAB/R2022b" CC=gcc
```

#### Using Octave
The Octave build of MEQ is still under development. As the MATLAB-provided versions of BLAS and LAPACK are not available, one should define the BLASLIB variable to CBLAS or MKL.

The following command allows to build the necessary MEX-files for `meq`:
```
make USE_OCTAVE=1 CC=gcc BLASLIB=CBLAS tbx
```

Note that neither the test suite nor the Simulink models can be used within Octave.

### Testing
Run the tests:
```
run_meq_tests(mytest)
```

Some `mytest` options:

* `'unit'` (run unit tests)
* `'integration'` (run integration tests)
* `'TCV'` (run tests on various TCV equilibria, requires TCV data access)
* `'CREATE'` (run tests on CREATE-supplied ITER scenarios, requires equilibrium files to be present)
* `'fgs'`,`'fge'`,`'fbt'`,`'liu'` (code specific tests)

## Getting started
* See `documentation/Getting_started.m` for an overview of help options.
* Type `doc meq` to open `meq` documentation
* See `documentation/tutorials` for `.m` files of tutorials

### TCV example calls (requires data access or running on SPC's lac clusters)
* Equilibrium reconstruction for TCV:
```
[L,LX,LY] = liuqe(shot,time);
```
* Solve static inverse problem for TCV:
```
 [L,LY] = fbt('tcv',shot); 
```
* Solve forward Grad-Shafranov equation TCV:
```
 [L,LX,LY] = fgs('tcv',shot,time); 
```

## Coordinate system
MEQ uses the coordinate system defined by `COCOS=17` [5]
This means: 

* Right-handed $(R,phi,Z)$ cylindrical coordinate system
* Right-handed $(\rho,\theta,phi)$ toroidal coordinate system 
* Polidal flux [in Wb] defined as $\psi = \int B_z dS_z$.


## Contributing/troubleshooting
* Flag bugs/feature requests here: [https://gitlab.epfl.ch/spc/tcv/tbx/meq/issues](https://gitlab.epfl.ch/spc/tcv/tbx/meq/issues)
* To work on an issue, open a merge request with corresponding branch.
* For more information on how to contribute, see the [GIT](https://spcwiki.epfl.ch/wiki/GIT) and [GitLab](https://spcwiki.epfl.ch/wiki/GIT/GitLab) page on SPCwiki

## License:
A license is required to use or modify any part of code. Please contact the code maintainers for licensing options.

## References
* [1] F. Hofmann and G. Tonetti, "Tokamak equilibrium reconstruction using faraday rotation measurements," Nucl. Fusion, vol. 28, no. 10, p. 1871, 1988.
* [2] F. Hofmann, "FBT - a free-boundary tokamak equilibrium code for highly elongated and shaped plasmas," Comput. Phys. Commun., vol. 48, no. 2, pp. 207-221, 1988.
* [3] F. Hofmann and G. Tonetti, "Fast identification of plasma boundary and X-points in elongated tokamaks," Nucl. Fusion, vol. 28, no. 3, pp. 519-522, Mar. 1988.
* [4] J.-M. Moret, B. P. Duval, H. B. Le, S. Coda, F. Felici, and H. Reimerdes, "Tokamak equilibrium reconstruction code LIUQE and its real time implementation", Fusion Eng. Des., vol. 91, no. 0, pp. 1-15, Feb. 2015.
*﻿[5] O. Sauter and S. Y. Medvedev, “Tokamak coordinate conventions:,” Comput. Phys. Commun., vol. 184, no. 2, pp. 293–302, Feb. 2013.
