!
! The code calculates the time autocorrelation function of a sequence of data
! Output: normalized time autocorrelation function as a function of time
! (cf. the documentation)
!
! Author: Giacomo Miceli
! EPFL, November 2012  
!
program time_corr
 implicit none
 integer :: i,j,k
 integer :: nstep   ! total step in the trajectory file
 integer :: istep   ! stochastic variable counter
 integer :: nsamp   ! nsamp*dt time between two calculated values of the correlation function
 integer :: timemax ! max number of points for the correlation function
 integer :: tmax    ! timemax/nsamp
 integer :: t0max   ! max number of t0 for the ensemble average
 integer :: isamp   ! correlation function counter
 integer :: it0     ! time interval between two t0
 integer :: t0,t
 integer :: delt

 integer, dimension (:), allocatable :: time0 ! vector of time 0 index
 integer, dimension (:), allocatable :: ntime

 real :: dt         ! time step of the stochastic variable sampling
 real :: dtime      ! time step of the correlation function
 real :: time
 real :: maxtime
 
 real :: A,Am       ! stochastic variable and its average
 real, dimension (:), allocatable :: A0
 real, dimension (:), allocatable :: corr
 real :: x
 
 character(256) :: inputfile

 call read_input
 
 tmax=int(REAL(timemax)/REAL(nsamp)) 
 t0max=int(REAL(nstep)/REAL(it0))
 isamp=0
 t0=0
 dtime=dt*nsamp
 
 allocate(A0(t0max), time0(t0max), ntime(tmax-1), corr(tmax))
 ntime=0
 
 open(11,file=inputfile,status='old')
 
 Am=0.0
 do i=1,nstep
 read(11,*)A
 Am=Am+A
 enddo
 Am=Am/real(nstep)
 rewind(11)

 isamp=0
 corr=0
 do istep=1,nstep
   
   read(11,*)A
   if(MOD((istep-1),nsamp).eq.0)then
     isamp=isamp+1                  
     if ( MOD(isamp-1,it0) .eq. 0 )then
       t0=t0+1
       if (t0 .le. t0max) then ! store A0 variables
         time0(t0)=isamp
         A0(t0)=A
       endif
     endif
    
     do t=1,MIN(t0,t0max)
       delt=isamp-time0(t)+1
       if (delt .lt. tmax) then
         ntime(delt) = ntime(delt)+1
         corr(delt) = corr(delt) + (A-Am)*(A0(t)-Am)
       endif
     enddo
   endif
 enddo
 
 write(*,*)"## normalized time autocorrelation function"
 do i=1,tmax-1
   time=dtime*(i-1)
   corr(i)=corr(i)/(REAL(ntime(i)))
   write(*,*)time,corr(i)/corr(1)
 enddo

 deallocate(A0, time0, ntime, corr)

!###############################################################################

 contains
 
!-------------------------------------------------------------------------------
!   READ input file
!-------------------------------------------------------------------------------

 subroutine read_input
 
 namelist/correlation/&
         nstep,timemax,nsamp,it0,dt,inputfile
 
! default parameters
 nstep=100000
 timemax=1000
 nsamp=1
 it0=10
 dt=0.02
 inputfile="output"
 
 read(*,nml=correlation,end=200,err=800)
 
 if (it0 .ge. nstep/2)then
 it0=nstep/2
 write(0,*)'WARNING: average on different t0s is not performed: it0 => nstep/2'
 endif
 if(nsamp .ge. nstep/2)then
 write(0,*)'STOP: isamp has to be << nstep/2'
 stop
 endif
 
 return   !  successful exit
200 continue
   write(0,*)'Read_Input: FATAL: premature end-of-file in standard input'
   stop
800 continue
   write(0,*)'Read_Input: FATAL: read error in standard input'
   stop
 
 end subroutine read_input

end program
