function fig = gausview(x,mu,sigma,title,rangeX,rangeY);

% GAUSVIEW 2D Gaussian densities exploration tool
%
%   Launch it with GAUSVIEW(X,MU,SIGMA,TITLE) where X is a N*2 matrix of
%   2-dimensional samples (one sample per line), MU is the mean of the
%   density, SIGMA is the covariance matrix of the density, and TITLE is the
%   (optional) title of the plot.
%
%   Buttons allow you to switch between 2D and 3D view, to edit the axes,
%   and to switch on/switch of the different views of the data (cloud of
%   points, mesh and contour plot).
%

if nargin < 2, mu = mean(x); end;
if nargin < 3, sigma = cov(x); end;
if nargin < 4, title = ''; end;
if nargin < 5, rangeX = 0:20:2000; end;
if nargin < 6, rangeY = 0:20:2000; end;

if isempty(title),
  fig = figure;
else,
  fig = figure('name',title);
end;

%colordef none;

% Make plots
set(gca,'xlim', [rangeX(1) rangeX(end)], ...
        'ylim', [rangeY(1) rangeY(end)], ...
        'zlim', [0 2.5e-5], ...
	'dataAspectRatio', [1 1 1e-8] );
pos = get(gca,'position'); pos(1) = pos(1) - 0.1; set(gca,'position',pos);
hold on;
grid on;
rotate3d on;
% - cloud of points
hpts = plot(x(:,1),x(:,2),'y+');
% - Gaussian mesh
[xx,yy] = meshgrid(rangeX,rangeY);
zz = gausspdf( [xx(:) yy(:)], mu, sigma );
zz = reshape(zz, size(xx));
hm = mesh(xx,yy,zz,'facecolor','none');
colormap(jet);
%set(hm,'visible','off');
% - Gaussian contour
[c,hc] = contour3(xx,yy,zz,'r-');
set(hc,'linewidth',2,'visible','off');

% Buttons
btnX = 0.78;
btnWdth = 0.22;
btnHt = 0.1;

cstr = [ 'if get(gco,''value''), ' ...
         'set(get(gco,''userdata''),''visible'',''on'');' ...
         'else;' ...
         'set(get(gco,''userdata''),''visible'',''off'');' ...
         'end;' ...
      ];
hb(1) = uicontrol('style','radiobutton', ...
    'units','normalized', ...
    'position', [btnX 0.85 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.85 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','View data points', ...
    'callback', cstr, ...
    'userdata', hpts, ...
    'value', 1, ...
    'visible', 'on');

hb(2) = uicontrol('style','radiobutton', ...
    'units','normalized', ...
    'position', [btnX 0.75 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.85 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','View pdf mesh', ...
    'callback', cstr, ...
    'userdata', hm, ...
    'value', 1, ...
    'visible', 'on');

hb(3) = uicontrol('style','radiobutton', ...
    'units','normalized', ...
    'position', [btnX 0.65 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.85 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','View pdf contour', ...
    'callback', cstr, ...
    'userdata', hc, ...
    'value', 0, ...
    'visible', 'on');


hb(4) = uicontrol('style','push', ...
    'units','normalized', ...
    'position', [btnX 0.5 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.85 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','2D view', ...
    'callback', 'view(2);', ...
    'visible', 'on');

hb(5) = uicontrol('style','push', ...
    'units','normalized', ...
    'position', [btnX 0.4 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.85 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','3D view', ...
    'callback', 'view(3);', ...
    'visible', 'on');


hb(6) = uicontrol('style','push', ...
    'units','normalized', ...
    'position', [btnX 0.25 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.85 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','Edit axes', ...
    'callback', 'axlimdlg;', ...
    'visible', 'on');


hb(7) = uicontrol('style','push', ...
    'units','normalized', ...
    'position', [btnX 0.05 btnWdth btnHt], ...
    'backgroundcolor', [0.8 0.8 0.8], ...
    'foregroundcolor', [0 0 0], ...
    'string','Close', ...
    'callback', 'close(gcf);', ...
    'visible', 'on');
