package scalashop

import scalashop.image.*

import java.awt.Dimension
import java.awt.Graphics
import java.awt.image.BufferedImage
import java.io.FileInputStream
import java.io.InputStream
import javax.imageio.ImageIO
import javax.swing.JComponent

class PhotoCanvas extends JComponent:

  private var imagePath: Option[String] = None
  private var imageScale: Int = 1

  var image: Image = loadScalaImage()

  override def getPreferredSize: Dimension =
    Dimension(image.width, image.height)

  private def loadScalaImage(): Image =
    val stream =
      this.getClass.getResourceAsStream("/scalashop/gameoflife/glider.bmp")
    try
      loadImage(stream)
    finally
      stream.close()

  private def loadFileImage(path: String): Image =
    val stream = FileInputStream(path)
    try
      loadImage(stream)
    finally
      stream.close()

  private def loadImage(inputStream: InputStream): Image =
    val bufferedImage = ImageIO.read(inputStream)
    Image.from(bufferedImage)

  def reload(): Unit =
    // do we have a known user-chosen path for the image? else load default
    image = imagePath match
      case Some(path) => loadFileImage(path)
      case None       => loadScalaImage()
    repaint()

  def loadFile(path: String): Unit =
    imagePath = Some(path)
    reload()

  def setScaling(scale: Int): Unit =
    imageScale = scale
    repaint()

  def applyFilter(filterName: String): Unit =
    val filteredImage = filterName match
      case "game-of-life-step" =>
        GameOfLife.Step(image)
      case _ => image

    image = filteredImage.build

    // for a somewhat accurate time measurement, avoid an async repaint
    paintImmediately(0, 0, image.width * imageScale, image.height * imageScale)

  override def paintComponent(gcan: Graphics): Unit =
    super.paintComponent(gcan)
    val width = image.width * imageScale
    val height = image.height * imageScale
    val bufferedImage =
      BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB)

    for
      x <- 0 until width
      y <- 0 until height
    do bufferedImage.setRGB(x, y, image(x / imageScale, y / imageScale))

    gcan.drawImage(bufferedImage, 0, 0, null)
