import SoapySDR
from SoapySDR import *  # SOAPY_SDR_ constants
import numpy as np

import usrpc


def get_available_devices(driver):
    return SoapySDR.Device.enumerate(f'driver={driver}')


def sdr_create_device(driver, serial):
    print(f"Creating {driver} device with serial {serial}")
    return SoapySDR.Device(f'driver={driver}, serial={serial}')

    
def sdr_setup_rx(driver, channel, config):
    """
    Set parameters for the RX channel
    - Uses values from the configuration object
    - Modifies the SDR object in place

    :param driver: the SDR object
    :param channel: the channel to configure (usually 0)
    :param config: the configuration object
    :return: None
    """

    # Use automatic DC offset correction
    # (It seems that this is not actually working - to further investigate)
    driver.setDCOffsetMode(SOAPY_SDR_RX, channel, True)

    # Set the Local Oscillator Offset (LOO) to remove the DC leakage
    # (Use soapy kwargs format, which is a richer version of python dicts)
    soapy_args = SoapySDR.KwargsFromString(f"OFFSET={config.LOO}")

    driver.setFrequency(SOAPY_SDR_RX, channel, config.fc, soapy_args)
    driver.setMasterClockRate(config.clockRateRx)
    driver.setClockSource(config.clockInputSource)
    driver.setSampleRate(SOAPY_SDR_RX, channel, int(config.clockRateRx/config.decimationRx))
    driver.setGain(SOAPY_SDR_RX, channel, config.gainRx)


def sdr_setup_tx(driver, channel, config):
    """
    Set parameters for the TX channel
    - Uses values from the configuration object
    - Modifies the SDR object in place

    :param driver: the SDR object
    :param channel: the channel to configure (usually 0)
    :param config: the configuration object
    :return: None 
    """

    # Use automatic DC offset correction
    # (It seems that this is not actually working - to further investigate)
    driver.setDCOffsetMode(SOAPY_SDR_TX, channel, True)

    # Set the Local Oscillator Offset (LOO) to remove the DC leakage
    # (Use soapy kwargs format, which is a richer version of python dicts)
    soapy_args = SoapySDR.KwargsFromString(f"OFFSET={config.LOO}")

    driver.setFrequency(SOAPY_SDR_TX, channel, config.fc, soapy_args)
    driver.setClockSource(config.clockInputSource)
    driver.setSampleRate(SOAPY_SDR_TX, channel, int(config.clockRateTx/config.interpolationTx))
    driver.setGain(SOAPY_SDR_TX, channel, config.gainTx)

    
def sdr_transmit(sdr, tx_time_0, dataToTx):
    """
    Transmit the data using the SDR
    
    :param sdr: the SDR
    :param tx_time_0: The timestamp when the first sample should be transmitted (SDR clock in ns)
    :param dataToTx: the data to transmit
    """

    # Define the chunk size for transmission
    chunk_size = int(usrpc.samplesPerFrameRx)

    # Timeout duration: 5s >> stream time
    timeoutUs = int(5e6) 
    
    # Setup TX stream for channel 0
    tx_stream = sdr.setupStream(SOAPY_SDR_TX, SOAPY_SDR_CF32, [0])
    status = sdr.activateStream(tx_stream)

    if status != 0:
        raise Exception('activateStream failed %s'%str(status))
    else:
        print("[TX] INIT OK!")
        print("[TX] Transmitting samples starting at: ", tx_time_0)

    # First chunk is scheduled for the transmit time
    # Rest is sent right after
    first_chunk = dataToTx[:chunk_size]
    rest_TX = dataToTx[chunk_size:]

    # Configure first flag to match timestamp with RX
    flags = SOAPY_SDR_HAS_TIME
    status = sdr.writeStream(tx_stream, [first_chunk], first_chunk.size, flags, tx_time_0)
    print(f"[Tx] Chunk 1 / {dataToTx.size // chunk_size + 1} ({chunk_size} samples) sent")

    for i in range(0, rest_TX.size, chunk_size):
        # The chunk might be smaller than the chunk size, but this 
        # is intended, as the signal is repeated.
        chunk = rest_TX[i:i + chunk_size]
        
        # Just send the continuous data (after initial scheduling).
        status = sdr.writeStream(tx_stream, [chunk], chunk.size, timeoutUs=timeoutUs)
        
        # SDR replies with the number of bytes that are prepared for transmission
        if status.ret == chunk.size:
            print(f"[TX] Chunk {i//chunk_size + 2} / {(dataToTx.size // chunk_size) + 1}  ({chunk_size} samples) sent...")
        else:
            print(f"Samples sent so far: {i + chunk.size} of {dataToTx.size}")
            raise Exception('transmit failed %s'%str(status))

    print("[TX] Finished transmitting all samples")
    print("[TX] Total samples transmitted: ", len(dataToTx))
    print("[TX] Cleaning up...")

    # Tx complete - deactivate stream
    sdr.deactivateStream(tx_stream)
    sdr.closeStream(tx_stream)


def sdr_receive(sdr, rx_start_time, rx_samples_buff, total_length):
    """
    Receive the samples using the SDR
    
    :param sdr: the SDR object
    :param rx_start_time: the timestamp to start receiving samples (in ns) on SDR clock 
    :param rx_samples_buff: the buffer to store the received samples (reference to the buffer)
    :param total_length: the total number of samples to receive
    """

    # Samples counter (keep track of how many samples we have so far)
    rx_samps = int(0)

    # Set RX chunk size for one sample buffer
    chunk_size = int(usrpc.samplesPerFrameRx)

    # Timeout is set to 5s >> stream time
    timeout_us = int(5e6) 

    # Configure RX stream to start at rx_start_time
    rx_stream = sdr.setupStream(SOAPY_SDR_RX, SOAPY_SDR_CF32, [0])
    rx_flags = SOAPY_SDR_HAS_TIME 
    status = sdr.activateStream(rx_stream, rx_flags, rx_start_time)

    print("[RX] Will receive %d samples in total"%total_length)
    if status != 0:
        raise Exception('activateStream failed %s'%str(status))
    else:
        print("[RX] INIT OK!")
        print("[RX] Receiving samples starting at: ", rx_start_time)

    # Receive the data in contiguous buffers
    while rx_samps < total_length:
        # The number of samples that we want to receive
        samples_to_receive = min(chunk_size, total_length - rx_samps)

        # Allocate empty buffer for the chunk
        rx_chunk_buff = np.zeros(samples_to_receive, dtype=np.complex64)
        status = sdr.readStream(rx_stream, [rx_chunk_buff], samples_to_receive, timeoutUs=timeout_us)

        # Sanity check: underflow occurred
        if (status.flags & SOAPY_SDR_HAS_TIME) == 0:
            raise Exception('receive fail - no timestamp on first readStream %s'%(str(status)))

        # accumulate buffer or exit loop
        if status.ret > 0:
            # print(f"RX BUFFS SIZE: {rx_samples_buff.size}, RX_BUFF SIZE: {rx_chunk_buff.size}, RX BUFFS SHAPE: {rx_samples_buff.shape}, RX_BUFF SHAPE: {rx_chunk_buff.shape}")
            print(f"[RX] Received: {rx_samps // chunk_size + 1} / {total_length // chunk_size} chunks ({status.ret} expecting {samples_to_receive} samples)...")

            # Copy the received samples to the buffer
            rx_samples_buff[rx_samps:rx_samps + status.ret] = rx_chunk_buff[:status.ret]

            rx_samps += status.ret
        else:
            print(f"[RX] failed to read buffer")
            print(f"[RX] received so far: {rx_samps} / {total_length}")
            print(f"[RX]status: {status}")
            break

    print(f"[RX] Finished receiving {rx_samps} samples in total")
    print(f"[RX] Cleaning up...")

    # Deactivate stream - cleanup
    sdr.deactivateStream(rx_stream)
    sdr.closeStream(rx_stream)
