function [dataOut] = usrpTxRx(samplesIn)
% USRP transceiver

global usrpc;

if isempty(usrpc)
    usrpConfig();
end

lengthDataOut = 4*numel(samplesIn);

fprintf(1,'Tx/Rx over USRP, duration of %g [seconds].\n', usrpc.txRxDuration);
% repeat the data such that we have a Tx time of usrpc.txRxDuration
noRepData = ceil(usrpc.txRxDuration/(usrpc.Ts*length(samplesIn)));
noRepData = max(10, noRepData); % to be sure we repeat it at least 10 times
fprintf(1,'Repeating the data %i times.\n', noRepData);

dataToTx = repmat(samplesIn, 1, noRepData);

%% USRP settings

% find connected radios
connectedRadios = findsdru

% if no boards are detected, we throw an error
if isempty(connectedRadios.Platform)
    error('No SDR boards detected. Please check if the boards are properly connected and try again.');
end

% if more than one board connected, 1st one is the Tx, 2nd the Rx;
% otherwise the same board is used for both Tx and Rx
if length(connectedRadios) > 1
    [usrpBoardPlatformTx, usrpBoardPlatformRx] = connectedRadios.Platform;
    [usrpBoardSerialNumTx, usrpBoardSerialNumRx] = connectedRadios.SerialNum;
    fprintf(1, ['Using two boards:\nTx is (platform) %s,' ...
        ' (serial number) %s.\nRx is (platform) %s,' ...
        ' (serial number) %s.\n'],...
        usrpBoardPlatformTx, usrpBoardSerialNumTx, usrpBoardPlatformRx,...
        usrpBoardSerialNumRx);
else
    usrpBoardPlatformTx = connectedRadios.Platform;
    usrpBoardPlatformRx = usrpBoardPlatformTx;
    usrpBoardSerialNumTx = connectedRadios.SerialNum;
    usrpBoardSerialNumRx = usrpBoardSerialNumTx;
    fprintf(1, ['Using one board: Tx/Rx is (platform) %s,' ...
        ' (serial number) %s.\n'],...
        usrpBoardPlatformTx, usrpBoardSerialNumTx);
end

% configure the Tx object
tx = comm.SDRuTransmitter(...
    'Platform',usrpBoardPlatformTx, ...
    'CenterFrequency',usrpc.fc, ...
    'LocalOscillatorOffset',usrpc.LOO,...
    'InterpolationFactor',usrpc.interpolationTx, ...
    'MasterClockRate', usrpc.clockRateTx,...
    'SerialNum', usrpBoardSerialNumTx,...
    'Gain', usrpc.gainTx,...
    'ClockSource', usrpc.clockInputSource,...
    'UnderrunOutputPort', true)

TxInfo = info(tx)

% configure the Rx object
rx = comm.SDRuReceiver(...
    'Platform',usrpBoardPlatformRx, ...
    'CenterFrequency',usrpc.fc, ...
    'LocalOscillatorOffset', usrpc.LOO,...
    'MasterClockRate', usrpc.clockRateRx,...
    'SerialNum', usrpBoardSerialNumRx,...
    'DecimationFactor', usrpc.decimationRx,...
    'Gain', usrpc.gainRx,...
    'SamplesPerFrame', usrpc.samplesPerFrameRx,...
    'ClockSource', usrpc.clockInputSource,...
    'OutputDataType', usrpc.OutputDataTypeUSRP,...
    'OverrunOutputPort', true)

RxInfo = info(rx)

% Tx/Rx

noFrames = floor(length(dataToTx)/rx.SamplesPerFrame);
dataRx = zeros(1, noFrames*rx.SamplesPerFrame);
l = 1; w = 1;
while l <= noFrames
    [dataRxUSRP, len] = rx();
    % fprintf(1,'While loop %i. \n', w); % to check how much is waiting
    % for the rx() to receive something
    w = w+1;
    if len > 0
        dataRx(1+(l-1)*rx.SamplesPerFrame:l*rx.SamplesPerFrame) = dataRxUSRP.';
        dataTxUSRP = tx(dataToTx(1+(l-1)*rx.SamplesPerFrame:l*rx.SamplesPerFrame).');
        %t = t + tFrame;
        fprintf(1, 'Tx frame %i out of %i. \n', l, noFrames);
        l = l+1;
    end
end

% release the objects
release(tx)
release(rx)

% dataRxUSRP = dataRxUSRP - mean(dataRxUSRP); % remove DC offset
% dataRxUSRP = dataRxUSRP/var(dataRxUSRP); % set to var 1

% scatter plot to check if no ADC saturation happened
scatterplot(dataRx);
grid on; title('Received Sequence');

% drop some data to avoid the transients at the end
stopData = length(dataRx) - floor(length(samplesIn)/2);
startData = stopData - lengthDataOut + 1;
dataRx1 = dataRx(startData:stopData);

figure;
hax=axes;
plot(abs(dataRx)); grid on; title('Rx data: vertical bars show selected data')
hold on; xlabel('Time'); ylabel('Magnitude');
VL1 = startData;
VL2 = stopData;
line([VL1 VL1],get(hax,'YLim'), 'Color', [1 0 0]);
line([VL2 VL2],get(hax,'YLim'), 'Color', [1 0 0]);

% plot the spectrum of the Rx data
fso = 1/usrpc.Ts; N = length(dataRx);
freqLineo = [-fso/2:fso/N:fso/2-fso/N];
figure;
plot(freqLineo, fftshift(abs(fft(dataRx))));
grid on; xlabel('f [Hz]'); ylabel('Magnitude'); title('OFDM Rx Spectrum');

%dataRx = dataRx - mean(dataRx);
dataOut = dataRx1;

end
