function tx_signal = transmitter(data_bits)
%TX_SIGNAL = TRANSMITTER(DATA_BITS) converts data bits to
%TX signal, using the parameters defined in OFDMC.
%
% To this end, the following steps are taken:
%   1. Data bits are mapped to constellation symbols.
%
%   2. Using the function OFDM_TX_FRAME_WITH_PILOTS data symbols are
%      converted to time-domain OFDM symbols with pilot symbols on the
%      selected carriers (and desired PSD mask).
%
%      The resulting time-domain signal is scaled to be normalized 
%      between -1 and +1, so as to avoid clipping by the USRP board.
%
%   3. The samples of time-domain data signal are prepended with a 
%      PREAMBLE that helps the receiver to detect the start of 
%      transmission and correct the CFO.  (The PREAMBLE is normalized
%      between -1 and +1.) 
%      
%      We also insert some "zero"s before data signal.  The receiver
%      uses these zeros to estimate the noise variance (needed for MMSE
%      estimation of channel coefficients).
 

global ofdmc;
if isempty(ofdmc)
    ofdmConfig(); 
end

function_mapper_usrp; % initializes function handles

if ~isvector(data_bits)
    warning('ofdm:reshaping','DATA_BITS is not a vector, reshaping it');
end

data_bits = data_bits(:);

if any(data_bits ~= 0 & data_bits ~=1)
    error('ofdm:invalidInput','DATA_BITS does not contain bits');
end

if numel(data_bits) < ofdmc.nDataBits
    %data_bits = [data_bits(:); zeros(ofdmc.nDataBits-numel(data_bits),1)];
    % Nicolae: add random bits instead of zeros
    randomPaddingBits = randi([0 1], ofdmc.nDataBits-numel(data_bits),1);
    data_bits = [data_bits(:); randomPaddingBits];
elseif numel(data_bits) > ofdmc.nDataBits
    warning('ofdmc:inputTooLong',['DATA_BITS contains more than %d bits.' ...
        ' Truncating it!'], ofdmc.nDataBits);
    data_bits = data_bits(1:ofdmc.nDataBits);
end

% 1. Modulate the data bits to constellation symbols

tx_bits = transpose(reshape(data_bits,log2(ofdmc.M),[])); % reshape over rows (for compatibility with Python)
decimal_symbols = sol_bi2de(tx_bits); % works over rows

%    get the mapping
switch(lower(ofdmc.constellationType))
    case 'qam'
        mapping = sol_qamMap(ofdmc.M);
    case 'psk'
        mapping = sol_pskMap(ofdmc.M);
    otherwise
        error('ofdm:invalidModulation','Unsupported modulation type %s', ...
            ofdmc.modtype);
end
%     normalize constellation to unit average energy - not really necessary
mapping = mapping./sqrt(mean(abs(mapping).^2));

%     modulator
data_symbols = sol_encoder(decimal_symbols, mapping);

% 2. Convert data symbols to time-domain OFDM symbols
tx_signal = ofdm_tx_frame_with_pilots(ofdmc.nCarriers, ...
    ofdmc.psdMask, ofdmc.cpLength, ...
    ofdmc.trainingSymbols, ...
    ofdmc.pilotIndices, ofdmc.pilotSymbols, data_symbols);

%    make sure the signal is scaled between +1 and -1
tx_signal = tx_signal / max(abs(tx_signal));


% 3. Prepend the preamble for symbol synchronization and CFO estimation.
% Insert zeros for noise estimation.
tx_signal = [ofdmc.preamble(:); ...                          % preamble
    zeros(ofdmc.nZeros,1); ...                               % zeros
    tx_signal(:)];                                           % data

% NOTE. ofdmc.preamble is normalized such that to fit between -1 and +1

tx_signal = tx_signal(:).';  % to make sure we have row vectors

if (ofdmc.verbose)
    % plot the frequency and time domain TX signals
    tfplotReImPhase(tx_signal,1/ofdmc.T,'s','TX Signal');
end

end

