function tx_symbols = sol_ofdm_tx_frame_with_pilots(num_carriers, psd_mask, prefix_length, training_symbols, pilot_indices, pilot_symbols, data_symbols)
%OFDM_TX_FRAME_WITH_PILOTS Generates an OFDM frame
%   TX_SYMBOLS = OFDM_TX_FRAME_WITH_PILOTS(NUM_CARRIERS, PSD_MASK, ...
%       PREFIX_LENGTH, TRAINING_SYMBOLS, PILOT_INDICES, PILOT_SYMBOLS, ...
%       DATA_SYMBOLS)
%
%       NUM_CARRIERS: number of carriers per OFDM block (power of 2) 
%       PSD_MASK: A {0,1}-valued vector of length NUM_CARRIERS, used to
%         turn off individual carriers.
%       PREFIX_LENGTH: cyclic-prefix length (in number of samples). 
%       TRAINING_SYMBOLS: vector of symbols known to the receiver, used
%         to estimate the channel. Its length is the number of ones in the
%         PSD_MASK (one training symbol per non-off carrier). 
%       PILOT_INDICES: indices of pilot symbols. 
%       PILOT_SYMBOLS: vector of pilot symbols.   
%       DATA_SYMBOLS: vector of symbols to transmit (it
%         will be padded with zeros if the number of data
%         symbols is not a multiple of the number of useful carriers, 
%         where the number of useful carriers is the number of ones 
%         in the PSD_MASK minus the number of pilot symbols).
%   TX_SYMBOLS: A column vector containing the generated OFDM symbols,
%   corresponding to one OFDM frame with the training symbols transmitted
%   during the first OFDM block and the data transmitted in the subsequent
%   OFDM blocks.

if ~isscalar(num_carriers) || num_carriers < 0  || mod(num_carriers,1) ~= 0
    error('ofdm_tx_frame:dimensionMismatch', ...
        'NUM_CARRIERS must be a positive scalar integer');
end

if ~isscalar(prefix_length) || prefix_length < 0  || mod(prefix_length,1) ~= 0
    error('ofdm_tx_frame:dimensionMismatch', ...
        'PREFIX_LENGTH must be a positive scalar integer');
end

if ~ isvector(psd_mask)|| numel(psd_mask) ~= num_carriers
    error('ofdm_tx_frame:dimensionMismatch',...
        'PSD_MASK must be a vector of length %d', ...
        num_carriers);
end

if any(psd_mask ~= 0 & psd_mask ~= 1)
    error('ofdm_tx_frame:invalidMask',...
        'PSD_MASK must be {0,1}-valued');
end

if ~isvector(pilot_indices)
    error('ofdm_tx_frame:invalidPilots', ...
        'PILOT_INDICES must be a vector');
end

if any(pilot_indices < 1 | pilot_indices > num_carriers)
    error('ofdm_tx_frame:invalidPilots', ...
        'PILOT_INDICES out of range');
end


if ~isvector(pilot_symbols)
    error('ofdm_tx_frame:invalidPilots', ...
        'PILOT_SYMBOLS must be a vector');
end

if numel(pilot_symbols) ~= numel(pilot_indices)
    error('ofdm_tx_frame:invalidPilots', ...
        'PILOT_INDICES and PILOT_SYMBOLS should have the same size');
end
    
    

psd_mask = logical(psd_mask);

if any(psd_mask(pilot_indices) == 0)
    error('ofdm_tx_frame:maskedPilots', ...
        'Some pilot carriers are masked')
end
    

% Determine the number of useful carriers
num_useful_carriers = sum(psd_mask == 1);

if (numel(training_symbols) ~= num_useful_carriers)
    error('ofdm_tx_frame:dimensionsMismatch', ...
          'PREAMBLE must contain exactly %d symbols', ...
          num_useful_carriers);
end

% Determine the number of data carriers and padd the input data with zeros
% such that we obtain an integer number of OFDM blocks
num_data_carriers = num_useful_carriers - numel(pilot_indices);

num_data_symbols = numel(data_symbols);
num_ofdm_symbols = ceil(num_data_symbols/num_data_carriers);
num_symbols_padding = num_ofdm_symbols * num_data_carriers - num_data_symbols;

if (num_symbols_padding ~= 0)
    warning('ofdm_tx_frame:padding', 'Padding data symbols with zeros');
end

data_symbols = [data_symbols(:); zeros(num_symbols_padding, 1)];

num_ofdm_symbols = num_ofdm_symbols + 1; % we use one OFDM symbol for the preamble (to aid channel estimation)

% Build the data matrix: the first column is for the preamble, and the following for the data
B = zeros(num_useful_carriers, num_ofdm_symbols);
B(:,1) = training_symbols(:);

% Create a pilot mask from the pilot_indices and psd_mask
pilot_mask = zeros(1,num_carriers);
pilot_mask(pilot_indices) = 1;
pilot_mask = pilot_mask(psd_mask);
pilot_mask = logical(pilot_mask);

% Insert in B the data and the pilot symbols
B(~pilot_mask,2:end) = reshape(data_symbols(:), num_data_carriers, []);
B(pilot_mask,2:end)  = repmat(pilot_symbols(:),1,num_ofdm_symbols-1);

% Add the off-carriers according to the psd_mask
A = zeros(num_carriers, num_ofdm_symbols);
A(psd_mask,:) = B;

% Do the IFFT (MATLAB IFFT applied on a matrix returns a matrix with the IFFT of every column)
a0 = ifft(A, num_carriers);

% Before sending through the channel we have to add the cyclic prefix
a = [a0(num_carriers-prefix_length+1:num_carriers,:); a0];

% Serialize, so that the output is a column vector
tx_symbols = a(:);

