function [tau_estim, cfo_estim] = sol_estimateTauAndCFO(rx_symbols, preamble, T, cfo_range)
%ESTIMATETAUANDCFO estimates the starting time of the data symbols and the
%carrier frequency offset, by correlating the received signal with the preamble. 
%[TAU_ESTIM, CFO_ESTIM] = estimateTauAndCFO(RX_SYMBOLS, ...
%		PREAMBLE, T, CFO_RANGE)
%  
%   RX_SYMBOLS: the vector of received symbols.
%   PREAMBLE: the preamble sequence.  
%   T: the sampling period of data.
%   CFO_RANGE: vector of candidates CFO values, used to search for a 
%     coarse CFO estimate. The coarse estimate is then refined 
%     using a second order approximation.
%   TAU_ESTIM: the estimate of the starting time of the data symbols.
%   CFO_ESTIM: the estimate of the carrier frequency offset.

global ofdmc;

if isempty(ofdmc)
    ofdmConfig();
end

if ~isscalar(T) || T < 0
    error('ofdm:invalidArgument','T is not a positive scalar.')
end

if ~isvector(cfo_range) || ~isreal(cfo_range)
    error('ofdm:invalidArgument','CFO_RANGE must be a real vector');
end


rx_symbols = reshape(rx_symbols,1,[]);
preamble = reshape(preamble,1,[]);

t = (0:numel(rx_symbols)-1) * T;

inner_products = zeros(size(cfo_range));
taus           = zeros(size(cfo_range));
selectedIP     = -1e6; % for testing

% We first find the coarse CFO
for d = 1:numel(cfo_range)
    % correct for the tentative value of CFO
    data_cfo_corr = rx_symbols .* exp(-1i*2*pi*cfo_range(d)*t);
    
    % compute the correlation
    R = xcorr(data_cfo_corr, preamble);  % actual correlation
    R = R(length(data_cfo_corr):end);      % remove zero padding and ramp-up
    R = R(1:end-length(preamble)+1);  % remove ramp-down 
    
    [Rmax, tau] = max(abs(R)); % find the maximum and its position
    inner_products(d) = Rmax;
    taus(d)           = tau;
    
    % for testing & plotting
    if Rmax > selectedIP
        selectedIP       = Rmax;
        selectedIPvector = R;
    end
    
end

if (ofdmc.verbose)
    figure;
    subplot(2,1,1);
    plot(cfo_range,inner_products, '*-'); 
    grid on;
    title('IP vs CFO');
    xlabel('CFO [Hz]');
    subplot(2,1,2);
    time_scale = (0:numel(selectedIPvector)-1)*T;
    plot(time_scale,abs(selectedIPvector), '*-'); 
    grid on;
    title('Max IP vs time');
    xlabel('\tau [s]');
 end


% Get the maximum IP over the cfo_range
[~,d]     = max(inner_products);
cfo_estim = cfo_range(d);
tau_estim = taus(d);

if (ofdmc.verbose)
    fprintf('[estimateTauAndCFO] Estimated parameters:\n\tCFO (coarse):\t%g\n\tTau      :\t%g\n', ...
        cfo_estim, tau_estim);
end

% Now we refine the estimation of CFO by using the quadratic approximation, as
% we did in GPS. For now we choose the 3 points as cfo_estim + [-1 0 1]*fCorr, 
% where fCorr is the (constant) step used in cfo_range
 cfo_rangeRefine = cfo_estim + [-1 0 1]*ofdmc.fCorr;

inner_products = zeros(size(cfo_rangeRefine));
rx_preamble    = rx_symbols(tau_estim:tau_estim+length(preamble)-1);
t              = (0:numel(rx_preamble)-1) * T;
for d = 1:length(cfo_rangeRefine)
    rx_preamble_corr  = rx_preamble .* exp(-1j*2*pi*cfo_rangeRefine(d)*t);
    inner_products(d) = rx_preamble_corr*preamble';
end

% Find a, b, and c, such that
%    inner_product(cfo) = a * cfo^2 + b * cfo * c
% is a second-order approximation of the CFO

% abc is a length three vector that contains a, b, and c.
abc = [cfo_rangeRefine(:).^2 cfo_rangeRefine(:) ones(numel(cfo_rangeRefine),1)]\abs(inner_products(:));

% In order for the saddle point to be a maximum, the coefficient "a" has to
% be negative. Let us check that this is indeed the case.
if abc(1) >= 0
    error('ofdm:SyncLost', 'Unable to refine CFO estimate => Sync lost.');
end

cfo_estim = -abc(2)/(2*abc(1));

if (ofdmc.verbose)
    fprintf(1, '\tCFO (fine):\t%g\n\n', cfo_estim);
end
