function rx_symbols = channelSimulator(tx_symbols, snr, cfo, clockOffset)
%RX_SYMBOLS = CHANNELSIMULATOR(TX_SYMBOLS, SNR, CFO, CLOCKOSET)
%Simulates the effective channel seen between two USRP boards.
% 
% TX_SYMBOLS: symbols that form the TX signal (symbol rate OFDMC.T)
% SNR: signal to noise ratio [dB] (default 30dB).
% CFO: carrier frequency offset [Hz] (default 284 Hz, can be
%   any value between -1kHz to 1kHz).
% CLOCKOFFSET: sampling clock offset (in samples) between the transmitter
%   and receiver boards (default 3 samples, can be any value from 0 to 9
%   samples, where we assume an upsampling factor of 10 with respect to 
%   OFDMC.T).
% RX_SYMBOLS: symbols that form the RX signal (symbol rate OFDMC.T).

% To simulate the channel, we repeat dataIn 10 times, we add impairments
% (noise, CFO, clock offset, multipath), we cut the sequence at a random
% location and we output a sequence of length 4*length(TX_SIGNAL).


global ofdmc;
if isempty(ofdmc)
    ofdmConfig();
end

lengthDataOut = 4*numel(tx_symbols);

if (nargin < 2)
    snr         = 30;   % [dB]
end

if (nargin < 3)
    cfo         = 284;  % [Hz]
end

if (nargin < 4)
    clockOffset = 3;    % [samplles]
end


if ~isvector(tx_symbols)
    warning ('ofdm:channelSimulator', 'TX_SIGNAL is not a vector, reshaping it!');
    tx_symbols = reshape(tx_symbols,1,[]);
end

if ~isscalar(snr)
    error('ofdm:channelSimulator', 'SNR is not scalar.');
end

if ~isscalar(cfo) || ~isreal(cfo) || cfo < -1e3 || cfo > 1e3
    error('ofdm:channelSimulator', 'CFO must be a real scalar between -1kHz to 1kHz');
end



resampleFactor   = 10; % for introducing the fixed clock offset with granularity of 1/resampleFactor
jitterValue      = 0; % clock jitter. Keep to 0 for now.
clockMismatchPPM = 0; % [ppm]: clock drift between Tx/Rx (min value 22, for implementation issues)
droppedSamples   = 1234; % we drop some samples from the beginning of the sequence


if ~isscalar(clockOffset) || mod(clockOffset,1) ~= 0 || ...
        clockOffset < 0 || clockOffset >= resampleFactor
    error('ofdm:channelSimulator', ...
        'CLOCKOFFSET must be an integer scalar between 0 and %d', ...
        resampleFactor);
end

    


% repeat the data several times
noRepData = 10;
dataTx = repmat(tx_symbols, 1, noRepData);

% add CFO
dataTx = dataTx.*exp(1j*2*pi*cfo*(0:1:length(dataTx)-1)*ofdmc.T);

% multipath channel
h = -[-0.646648866062013  -0.413731206902691  -0.192749918053592];
%ofdmc.timeDomainChannel = zeros(1, ofdmc.cpLength);
%ofdmc.timeDomainChannel(1:length(h)) = h; % to compare with the estimated one

% normalize impulse response
%h = h/norm(h)

if (length(h) > ofdmc.cpLength+1)
    warning('ofdm:impulseResponseTooLong', ...
        'The channel impulse response is larger than the cyclic prefix length => it will create ISI');
end

% convolve the signal with channel impulse response
dataTx = conv(dataTx, h);

% Add noise
if snr < inf
    dataTx = awgn(dataTx, snr, 'measured');
    % Note: we need to use awgn(dataTx, snr-10*log10(USF), 'measured'); But
    % here USF is 1 (no upsampling) so the noise variance is correct!
end

% drop some samples
dataTx = dataTx(droppedSamples:end);

% resample and introduce some (fixed) clock offset / jitter
if clockOffset > 0
    puResampled = resample(dataTx, resampleFactor, 1);
    
    % introduce some jitter
    samplingMoments = 1+clockOffset:resampleFactor:length(puResampled)-resampleFactor;
    jitterV = randi([-jitterValue jitterValue], 1, length(samplingMoments));
    
    puReRe = puResampled(samplingMoments+jitterV);
    
else
    puReRe = dataTx;
end

% add some drifting clock 
if clockMismatchPPM > 0
    [N,D] = rat((1+clockMismatchPPM*1e-6), 1e-7);
    puReRe = resample(puReRe, N, D);
    %puReRe = resample(puReRe, N, 1); puReRe = downsample(puReRe, D);
end


dataRx = puReRe;

% select the data we output
stopData = length(dataRx) - floor(length(tx_symbols)/2);
startData = stopData - lengthDataOut + 1;
dataRx1 = dataRx(startData:stopData);

if (ofdmc.verbose)
    % plot the magnitude of the Rx data
    figure;
    hax=axes;
    plot(abs(dataRx)); grid on; 
    title('Rx data: vertical bars show selected data')
    hold on; 
    xlabel('Sample Index'); 
    ylabel('Magnitude');
    VL1 = startData;
    VL2 = stopData;
    line([VL1 VL1],get(hax,'YLim'), 'Color', [1 0 0]);
    line([VL2 VL2],get(hax,'YLim'), 'Color', [1 0 0]);
    
    % plot the spectrum of the Rx data
    fso = 1/ofdmc.T; 
    N = length(dataRx);
    freqLineo = [-fso/2:fso/N:fso/2-fso/N];
    figure;
    plot(freqLineo, fftshift(abs(fft(dataRx))));
    grid on; 
    xlabel('f [Hz]'); 
    ylabel('Magnitude'); 
    title('OFDM Rx Spectrum');
end

rx_symbols = dataRx1;

end
