% TEST_OFDMCHEST Script to test the OFDM channel estimation

clear all; close all; 
%clc

function_mapper_mmse;

% System parameters
num_carriers = 256;    % total number of carriers
num_zeros = 5;         % number of unsused carriers (at each end of the frequency spectrum)
prefix_length = 25;    % length of the cyclic prefix
num_ofdm_symbols = 10000; % number of OFDM symbols per frame (1 will be used to transmit the preamble and the rest is for data)
M_preamble = 4;        % we use 4-QAM for the preamble
M_data = 4;           % we use 4-QAM for the data

SNR = 20 % in dB

% Derived parameters
num_useful_carriers = num_carriers - 2 * num_zeros;

% Preamble and data constellation
constel_preamble = sol_qamMap(M_preamble);
constel_data = sol_qamMap(M_data);

% Transmitter
% Generate preamble and data
preamble = randi([0, M_preamble-1], [num_useful_carriers, 1]);
preamble_symbols = sol_encoder(preamble, constel_preamble);
%preamble_symbols=ones(size(preamble_symbols));

data = randi([0, M_data-1], [num_useful_carriers * (num_ofdm_symbols-1), 1]);
data_symbols = sol_encoder(data, constel_data);

% Create a simple mask that zeros out carriers at the end and the beginning
% of the spectrum
psd_mask = [zeros(1,num_zeros), ones(1,num_useful_carriers),  ...
    zeros(1,num_zeros)];
psd_mask = fftshift(psd_mask); % the positive frequencies come first!

% Generate OFDM signal to be transmitted
tx_signal = sol_ofdm_tx_frame(num_carriers, prefix_length, ...
    preamble_symbols, psd_mask, data_symbols);
E_tx = mean(abs(tx_signal).^2); % power of transmitted signal (var(tx_signal))

% Channel

% CP should be >= L-1, where L is the channel length; 
% (L-1) is the length of the channel history 
channel_length = prefix_length + 1; 

% random impulse response for test purposes
% h = 1/sqrt(2) * (randn(1,channel_length) + 1i*randn(1,channel_length)); % complex channel
% h = randn(1,channel_length); % real channel (again, for testing purposes)

% multipath channel described in class
amplitudes = [1.3290, 0.6754, 0.6516, -0.2432];
delays = [0 0.5 1.5 2];
Ka = eye(length(delays));

lengthTailsSinc = 10; % length of the tails for the sinc (reconstruction filter)
h = create_multipath_channel_filter(amplitudes, delays, lengthTailsSinc);

% simple channel 
% which just adds WGN (non frequency selective channel, i.e, no ISI).
% to use it, uncomment the following line
% h = [1 zeros(1,channel_length-1)]; 

% normalize impulse response
% h = h/norm(h)

if (length(h) > prefix_length+1)
     warning('test_ofdm:impulseResponseTooLong', 'The channel impulse response is larger than the cyclic prefix length => it will create ISI');
end

% convolve tx_signal with channel impulse response
rx_signal = conv(tx_signal, h);

% add AWGN
sigma = 10^(-SNR/20)*sqrt(E_tx);
noise = sigma/sqrt(2)*(randn(size(rx_signal)) + 1i*randn(size(rx_signal)));

rx_signal_noisy = rx_signal + noise;

% Receiver
Rf = sol_ofdm_rx_frame(rx_signal_noisy, num_carriers, psd_mask, prefix_length);
  
% Channel coefficients (lambdas) estimation

% h known
lambda1 = sol_channel_est1(num_carriers, psd_mask, h);

% delays and variances known
sigma2 = num_carriers*sigma^2; % the fft transform increases the noise variance!
lambdaMMSE = channel_estMMSE(Rf, num_carriers, psd_mask, preamble_symbols, Ka,...
    delays+lengthTailsSinc, sigma2); % note that the delays are shifted because of the sinc's tails

% no channel information known, we use the pedestrian approach of dividing
% the channel output by the preamble
lambdaLS = channel_estLS(Rf, preamble_symbols);

% We determine the variance of each estimate (or the error)
varMMSE = var(lambda1-lambdaMMSE.')
varLS = var(lambda1-lambdaLS.')


% Channel equalization, demodulation and determining SER
% The channel response is assumed to remain constant during the whole frame

eq_signal1 = Rf .* repmat(1./lambda1(:), 1, num_ofdm_symbols);
suf_statistics1 = eq_signal1(:);
estim_data1 = sol_decoder(suf_statistics1(length(preamble_symbols)+1:end), constel_data);
SER1 = sum(estim_data1 ~= data) / numel(data) %#ok<NOPTS>

% Debugging information
suf_data1 = suf_statistics1(length(preamble_symbols)+1:end);
figure; plot(suf_data1, '*b'); xlabel('Re'); ylabel('Im'); title('Received constellation after OFDM demodulation and equalization');
hold on; plot(constel_data, '+r'); grid on; legend('Received constellation', 'Transmitted constellation');

eq_signal2 = Rf .* repmat(1./lambdaMMSE, 1, num_ofdm_symbols);
suf_statistics2 = eq_signal2(:);
estim_data2 = sol_decoder(suf_statistics2(length(preamble_symbols)+1:end), constel_data);
SER_MMSE = sum(estim_data2 ~= data) / numel(data) %#ok<NOPTS>

eq_signal3 = Rf .* repmat(1./lambdaLS, 1, num_ofdm_symbols);
suf_statistics3 = eq_signal3(:);
estim_data3 = sol_decoder(suf_statistics3(length(preamble_symbols)+1:end), constel_data);
SER_LS = sum(estim_data3 ~= data) / numel(data) %#ok<NOPTS>

% for instructional purposes
figure;
plot(abs(lambda1),'-*b'); hold on;
plot(abs(lambdaMMSE),'-+g');
plot(abs(lambdaLS),'-r');
xlabel('Frequency index'); ylabel('Lambda estimates');
grid on; legend('lambda1','lambdaMMSE','lambdaLS');
