import numpy as np
from numpy import linalg as LA
from scipy import fft
from scipy.signal import upfirdn
from scipy.signal import fftconvolve
import matplotlib.pyplot as plt
import warnings


def ofdm_tx_frame(num_carriers, prefix_length, training_symbols, psd_mask, data_symbols):
    """
    Generates an OFDM frame.

    :param num_carriers: number of carriers per OFDM block (power of 2)
    :param prefix_length: cyclic-prefix length (in number of samples)
    :param training_symbols: vector of symbols known to the receiver,
        used to estimate the channel. Its length is the number of ones
        in the PSD_MASK (one training symbol per non-off carrier).
    :param psd_mask: A {0,1}-valued vector of length NUM_CARRIERS, used
        to turn off individual carriers.
    :param data_symbols: vector of symbols to transmit (it will be padded
        with zeros if the number of data symbols is not a multiple of the
        number of useful carriers)

    :return: tx_symbols: A column vector containing the generated OFDM symbols,
        corresponding to one OFDM frame with the training symbols transmitted
        during the first OFDM block and the data transmitted in the subsequent
        OFDM blocks.
    """



    return tx_symbols


def create_multipath_channel_filter(amplitudes, delays, L):
    """
    Creates the sampled response of multipath channel.

    We assume that the shaping pulse is a sinc. This is the reconstruction filter.
    DELAYS and AMPLITUDES are vectors of the same length, specifying the
    strength and delay of each path. The DELAYS must be specified relative
    to the sampling period.

    :param amplitudes: vector specifying the strengths of the paths
    :param delays: vector specifying the delays of the paths
    :param L: The length of the tails of the sinc

    :return: h: contains the samples of filtered impulse response.
    """

    if amplitudes.size != delays.size:
        raise ValueError('create_multipath_channel_filter:wrongInputDimensions', 'AMPLITUDES and DELAYS must be vectors of the same length')

    # Here we implement the formula of h_overall, according to the lecture
    # notes (as well as in Hw exercise "Symbol-level channel" - for now we have
    # 1 sample per symbol, so this is the sample-level channel as well).
    # h_overall = p*h*q, with p,q being sinc functions.
    timeLine = np.arange(-L, L+1, 1)
    F = np.tile(timeLine, (delays.size, 1)).transpose()
    F = F - np.tile(delays, (timeLine.size, 1))
    h = np.dot(np.sinc(F), amplitudes)

    return h


def ofdm_rx_frame(rx_symbols, num_carriers, psd_mask, prefix_length):
    """
    Receiver for ODFM signals (without channel equalization).

    :param rx_symbols: vector of channel outputs. It is a filtered and noisy
        version of the transmitted symbols.
    :param num_carriers: number of carriers per OFDM block (power of 2). The
        same as in OFDM_TX_FRAME.
    :param psd_mask: A {0,1}-valued vector of length NUM_CARRIERS. The same
        as in OFDM_TX_FRAME.
    :param prefix_length: cyclic-prefix length (in number of samples). The
        same as in OFDM_TX_FRAME.

    :return: RF: Matrix of TRAINING_SYMBOLS and DATA_SYMBOLS (see OFDM_TX_FRAME)
        as seen at the output of the equivalent "parallel channels" created by
        OFDM. To obtain the RF matrix, we start by rearranging the received
        symbols columnwise in a matrix that has (NUM_CARRIERS + PREFIX_LENGTH)
        rows. In doing so, we remove the tail of RX_SYMBOLS that don't fill an
        entire column. Then we remove the cyclic prefix and take the FFT.
        Finally, the symbols that correspond to unused carriers are removed.
    """



    return Rf


def channel_est1(num_carriers, psd_mask, h):
    """
    Takes the channel impulse response h and returns the
    channel coefficients in the frequency domain.

    :param num_carriers: number of carriers per OFDM block (FFT/IFFT size)
    :param psd_mask: The PSD Mask used by the receiver.
        A {0,1}-valued vector of length NUM_CARRIERS used to
        turn off individual carriers if necessary (e.g., so as to
        avoid interference with other systems)
    :param h: Channel impulse response.

    :return: LAMBDA_CHANNEL: Column vector containing channel coefficients in the
        frequency domain. The number of elements of LAMBDA equals the
        number of ones in PSD_MASK. (We do not care about channel gains
         on the carriers that are turned off.)
    """



    return lambda_channel
