function [rp, b, f] = solveRangeEquationsVia_Newton(sp, rho_c)
% [RP, B, F] = SOLVERANGEEQUATIONSVIA_NEWTON(SP, RHO_C) solves the receiver
% position equation system, given the satellite position matrix SP and the
% corrected pseudorange vector RHO_C. The dimensions of SP are NUMSAT x 3,
% where NUMSAT is the number of visible and correctly decoded satellites;
% RHO_C is a row vector of NUMSAT elements.
% Output RP is the receiver position (row vector).
% Output B is the difference between the range and the pseudorange. 
% F is a row vector of NUMSAT elements. It contains norm(SP-RP)-rho_c-b. 
% In an ideal situation, F=0. It means that all equations can be satisfied
% exactly. Satellites for which the corresponding F-value is large should
% be disregarded, provided that we still have at least 4 satellites. 


global gpsc;

% Compute number of satellites, raise an error if there are not enough satellites
numsat = size(sp, 1);
if (numsat < 4)
    error('rcvrpos:solveRangeEquationsVia_Newton:NotEnoughSatellites', 'You need at least 4 satellites to compute the receiver position');
end

% Start with an initial guess
rp = zeros(1, 3);
b = 0;

% The idea is to use Newton's method for finding the x for which 
% f_k(x) = 0 for all k, where k is an index to the visible satellites, 
% x=[rp,b]and f_i(x)= norm(sp_i-rp)-rph_c(i)-b.
% Because there are more equations than variales, and because the equations
% are noisy, we have to settle for an x for which f_k(x) is almost zero for
% all k.
% There is a side note for the details

e = inf;
norm_old = inf;
e_stop = 1000;

while (abs(e) > e_stop) % stop if the norm of the solution does not change anymore    
    f = zeros(1,numsat);  % f=[f_1, ... ,f_numsat];
    J = zeros(4,numsat); % Jacobian
    for k = 1:numsat
        f(k) = norm(sp(k,:) - rp) - (rho_c(k) + b); % this is f(x^n)
        J(:,k) = [-(sp(k,:) - rp)' / norm(sp(k,:) - rp); -1]; % k-th column of the Jacobian
    end
    
    next = [rp, b] - f*pinv(J); % x^{n+1} = x^n - f(x^n) J^{-1}(x^n)
    rp = next(1:3);
    b = next(4);
    
    e = norm_old - norm(f); % stop if the norm of f(x^n) no longer changes
    norm_old = norm(f);
end

end % function solveRangeEquationsVia_Newton()
