function [subframes, subframesIDs] = sol_bits2subframes(bits)
%BITS2SUBFRAMES Returns a matrix containing the subframes
%   in the desired order
%   [SUBFRAMES, SUBFRAMESIDS] = BITS2SUBFRAMES(BITS) returns the matrix
%   SUBFRAMES having as its columns the subframes extracted from BITS.
%   The IDs of the subframes are returned in SUBFRAMESIDS.  BITS must
%   be a row vector containing a concatenation of subframes (0 and 1
%   elements) that have already been checked for parity.  Provided that
%   BITS is sufficiently long, SUBFRAMES contains the subframes with
%   id 1,2,3, in that order. (It might contain additional subframes 
%   and the first column is not necessarily subframe 1.)
%   These are the subframes that we use to obtain the ephemerides.


    global gpsc;
    if isempty(gpsc)
        gpsConfig();
    end	

    % Number of subframes per page
    SFPP = 5;

    % Check parameters: bits can only contain 0's and 1's, and must contain a multiple of bpsf bits. 
    if any((bits ~= 0) & (bits ~= 1))
        error('subframes2page:notBinaryInput', 'BITS can only contain 0''s and 1''s');
    end

    nsf = length(bits) / gpsc.bpsf;
    if (nsf ~= fix(nsf))
        error('subframes2page:notIntegralNumberOfSubframes', 'BITS must contain a multiple of %d bits', gpsc.bpsf);
    end

    % Nicolae: check if we have enough subframes. It can happen (from
    % removeExcessBits) that we have some subframes, but not enough. This
    % can also cause trouble if the first subframes as dropped, since we
    % get wrong info for computing the pseudorange
    if (nsf < SFPP + 3)
        error('LCM:NotEnoughSubframes', 'subframes2page: there are not enough subframes (%i out of 8), probably wrong bits...\n', nsf);
    end
    
    % Put in matrix form
    subframes = reshape(bits, gpsc.bpsf, nsf);
   
    % Extract SFID fields and convert to decimal.
    sfids_b = subframes(50:52, :)';
    subframesIDs = bi2de(sfids_b, 'left-msb')';
    fprintf(1, 'Found subframes %s.\n', mat2str(subframesIDs));
    
end
