/*
 *  s632.cpp
 *  
 Programme to integrate a triple of (non-linear) population equations for BIO 341.
 
 dx/dt = f1(x,y,z)
 dy/dt = f2(x,y,z)
 dz/dt = f3(x,y,z)

 using a 4th order Runge-Kutta scheme. The functions f1, f2, f3 are coded below. Currently (09/11/23) the z component is unused.
 
 Note that to integrate a simple ODE like dy/dt = f(y), you use x as the time variable and increment it manually in the loop,
 and y is the function to be integrated. The starting point has to be chosen accurately.
 
 *   
 */

#include <cmath>
#include <iostream>
#include <fstream>

// Useful type declarations

typedef std::istream                zInStream;
typedef std::ostream                zOutStream;
typedef std::ifstream               zInFileStream;
typedef std::ofstream               zOutFileStream;
typedef std::fstream                zFileStream;
typedef std::istringstream          zInStringStream;
typedef std::ostringstream          zOutStringStream;
typedef std::ios                    zIos;

#define zString                      std::basic_string<char>

using namespace std;

// Forward declaration

void Runge_Kutta(const double a[6], double dt, double xn, double yn, double zn, double* pxn1, double* pyn1, double* pzn1);


int main(int argc, char* argv[])
{
// ****************************************
    // Input and output file specs
	
	zString m_OutFileName;     // Name of file attached to the output stream

	bool m_IOSuccess;		// true = no error, false = error
	bool m_bOpenFlag;		// Flag showing whether the file is open or not

//    zInFileStream  m_inStream;
    zOutFileStream  m_outStream;

	// Check if the input data has been passed in via the command line: note that argv[0] = executable name; if not, we prompt for input
	
    long NSTEPS = 100;
    double x0 = 0.0;  // initial values
    double y0 = 0.0;
    double z0 = 0.0;

    double params[6];
    params[0] = 0.0;
    params[1] = 0.0;
    params[2] = 0.0;
    params[3] = 0.0;
    params[4] = 0.0;
    params[5] = 0.0;

            
	if(argc == 1)
	{
        // Input modified to integrate 3 equations simultaneously.
        
	    std::cout << "****************************************" << endl;
	    std::cout << "Enter file name, N, mu, and initial point x0, y0 z0" << endl;
	    std::cin >> m_OutFileName >> NSTEPS >> params[0] >> x0 >> y0 >> z0;
	}
	else
	{
        std::cout << "Error, only enter parameters when code prompts for them" << endl;
        return 1;
    }

	m_outStream.open(m_OutFileName.c_str());

	// If the file could not be opened flag an error

	if(m_outStream.fail())
	{
	    std::cout << "ERROR - unable to open output file " << m_OutFileName << endl;
		m_IOSuccess = false;
		m_bOpenFlag = false;
		return 1;
	}
	else
	{
		m_IOSuccess = true;
		m_bOpenFlag = true;
	}
// ****************************************
			
    double dt = 0.01;
    
    double xn, yn, zn;
    double xn1, yn1, zn1;
    
    xn = x0;
    yn = y0;
    zn = z0;

    for(long i=0; i<NSTEPS; ++i)
    {
        m_outStream << xn << " " << yn << " " << zn << endl;
        
        Runge_Kutta(params, dt, xn, yn, zn, &xn1, &yn1, &zn1);
        
        xn = xn1;
        yn = yn1;
        zn = zn1;
    }
	

// ****************************************
// Clean up
	
	m_outStream.close();

    return 0;
}

// **********************************************************************
// The trio of functions to return the time derivatives at a given time.
//
// These are the Lorenz equations from page 319 of Strogatz.

double f1(const double a[6], double x, double y, double z)
{
    return 10.0*(y - x);
}

double f2(const double a[6], double x, double y, double z)
{
    return a[0]*x - y - x*z;
}

double f3(const double a[6], double x, double y, double z)
{
    return x*y - 2.6667*z;
}


// **********************************************************************
// Function to integrate the two functions f1(x(t), y(y)), f2(x(t), y(t)) using 4th order Runge-Kutta by one time-step.
// It receives the current point (xn, yn) and returns the new point (xn1, yn1) as pointers.
// Note that the functions are assumed not to be explicitly time-dependent.
//
// It uses the two subsidiary functions f1(x,y), f2(x,y) to get the derivatives.
//
// k_ij = coefficient with i = 1, 2, 3, 4, and j = x, y, z.

void Runge_Kutta(const double a[6], const double dt, const double xn, const double yn, const double zn, double *pxn1, double *pyn1, double *pzn1)
{
    const double k11 = f1(a, xn,yn,zn)*dt;
    const double k12 = f2(a, xn,yn,zn)*dt;
    const double k13 = f3(a, xn,yn,zn)*dt;

    const double k21 = f1(a, xn + 0.5*k11,yn + 0.5*k12,zn + 0.5*k13)*dt;
    const double k22 = f2(a, xn + 0.5*k11,yn + 0.5*k12,zn + 0.5*k13)*dt;
    const double k23 = f3(a, xn + 0.5*k11,yn + 0.5*k12,zn + 0.5*k13)*dt;

    const double k31 = f1(a, xn + 0.5*k21,yn + 0.5*k22,zn + 0.5*k23)*dt;
    const double k32 = f2(a, xn + 0.5*k21,yn + 0.5*k22,zn + 0.5*k23)*dt;
    const double k33 = f3(a, xn + 0.5*k21,yn + 0.5*k22,zn + 0.5*k23)*dt;

    const double k41 = f1(a, xn + k31,yn + k32,zn + k33)*dt;
    const double k42 = f2(a, xn + k31,yn + k32,zn + k33)*dt;
    const double k43 = f3(a, xn + k31,yn + k32,zn + k33)*dt;

    *pxn1 = xn + 0.1666666*(k11 + 2.0*k21 + 2.0*k31 + k41);
    *pyn1 = yn + 0.1666666*(k12 + 2.0*k22 + 2.0*k32 + k42);
    *pzn1 = zn + 0.1666666*(k13 + 2.0*k23 + 2.0*k33 + k43);
}
